package com.horizon.wf.api;

import com.horizon.core.HorizonCore;
import com.horizon.wf.IWorkResult;
import com.horizon.wf.IWorkflowManager;
import com.horizon.wf.IWorkflowOperator;
import com.horizon.wf.WorkflowFactory;
import com.horizon.wf.bean.WorkParaBean;
import com.horizon.wf.config.HZResourceBundle;
import com.horizon.wf.config.PubInfo;
import com.horizon.wf.core.management.InstanceManagement;
import com.horizon.wf.core.mem.MemCacheUtil;
import com.horizon.wf.core.runmsg.RunMsgUtil;
import com.horizon.wf.core.work.ITaskWork;
import com.horizon.wf.core.work.TaskWork;
import com.horizon.wf.definition.FlowDBUtil;
import com.horizon.wf.definition.tools.NodeTypeEnum;
import com.horizon.wf.entity.db.*;
import com.horizon.wf.entity.user.impl.AuthUserImpl;
import com.horizon.wf.expand.impl.IAppInterface;
import com.horizon.wf.global.CollectionUtil;
import com.horizon.wf.global.StaticFunExtend;
import com.horizon.wf.global.StaticVarExtend;
import com.horizon.wf.global.StringUtilExtend;
import com.horizon.wf.plugins.PluginsUtil;
import com.horizon.wf.runmsg.RunMsgImpl;
import com.horizon.wf.tools.AccessUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * 流程管理操作类
 * @author liys
 * @version 1.0
 * @since hzwf v7.2
 *
 */
public class WorkflowManager extends WfmWorkList implements IWorkflowManager{
	private static final Logger LOGGER = LoggerFactory.getLogger(WorkflowManager.class);
	/**
	 * 工作移交(此方法只做移交标记,并不实际修改权限,实际权限更改在接收人接收后执行)
	 * @param list  参数map列表,包含以下key
	 * 		<br> key = workid 需要进行移交的工作列表(实例id)
	 * 		<br> key = trackid 实例对应的路径id
	 * 		<br> key = olduserid 移交人ID 带前缀
	 * 		<br> key = newuserid 接收人ID 带前缀
	 *  	<br> key = oldusername 移交人名称
	 * 		<br> key = newusername 接收人名称
	 * @param preservePermissions 是否保留权限
	 * @param tenantid 租户id
	 * @return 移交是否成功
	 */
	public boolean handoverWork(List<? extends Map<String,String>> list, boolean preservePermissions, String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return handoverWork(list,preservePermissions,tenantid,identifier);
	}
	/**
	 * 工作移交(此方法只做移交标记,并不实际修改权限,实际权限更改在接收人接收后执行)
	 * @param list  参数map列表,包含以下key
	 * 		<br> key = workid 需要进行移交的工作列表(实例id)
	 * 		<br> key = trackid 实例对应的路径id
	 * 		<br> key = olduserid 移交人ID 带前缀
	 * 		<br> key = newuserid 接收人ID 带前缀
	 *  	<br> key = oldusername 移交人名称
	 * 		<br> key = newusername 接收人名称
	 * @param preservePermissions 是否保留权限
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 移交是否成功
	 */
	public boolean handoverWork(List<? extends Map<String,String>> list, boolean preservePermissions,String tenantid,String identifier){
		Map<String,List<Object>> saveMap = new LinkedHashMap<String, List<Object>>();
		PubInfo pubInfo = PubInfo.getPubInfo(tenantid);
		for(Map<String,String> map:list){
			DBHandover handover = new DBHandover();
			handover.setId(StaticFunExtend.getUnid());
			handover.setNewname(map.get("newusername"));
			handover.setOldname(map.get("oldusername"));
			String olduserid = map.get("olduserid");
			String newuserid = map.get("newuserid");
			
			AuthUserImpl newAuth = AuthUserImpl.getAuthUserByStr(newuserid);
			handover.setNewuser(newAuth.getAuthId());
			handover.setNewuserid(newuserid);
			
			AuthUserImpl auth = AuthUserImpl.getAuthUserByStr(olduserid);
			handover.setOlduser(auth.getAuthId());
			handover.setOldsubjection(auth.getSubjectionId());
			handover.setOldsubjectiontype(auth.getSubjectionType());
			handover.setStatus("0");
			handover.setWorkid(map.get("workid"));
			handover.setPreservepermissions(preservePermissions?"1":"0");
			handover.setTrackid(map.get("trackid"));
			handover.getSQL(saveMap);
			
			Map<String,String> para = new HashMap<String,String>();
			para.put("workid", map.get("workid"));
			para.put("trackid",  map.get("trackid"));
			para.put("identifier", identifier);
			para.put("actionname",pubInfo.getInfo("Magr_Msg0001", "工作移交" ));
			para.put("actionUserid", handover.getOlduser());
			para.put("actionUsername", handover.getOldname());
			para.put("memo", pubInfo.getInfo("Magr_Msg0002","移交给[|name|]")
								.replace("|name|", handover.getNewname()));
			
			DBLog trackLog = StaticFunExtend.getActionCommon().makeDbLogForMgr(para);
			trackLog.getSQL(saveMap);
		}
		if(saveMap.isEmpty()){
			return false;
		}
		return AccessUtil.getInstance().executeMultiUpdate(saveMap, identifier);
	}
	/**
	 * 接收工作移交
	 * @param list 参数DBHandover列表
	 * @param tenantid 租户id
	 * @return
	 */
	public Map<DBHandover,String> receiveHandoverWork(List<DBHandover> list, String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return receiveHandoverWork(list,tenantid,identifier);
	}
	/**
	 * 接收工作移交
	 * @param list 参数DBHandover列表
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return
	 */
	public Map<DBHandover,String> receiveHandoverWork(List<DBHandover> list,String tenantid, String identifier){
		 Map<DBHandover,String> resultMap = new HashMap<DBHandover,String>();
		 for(int i=0,n=list.size();i<n;i++){
			DBHandover handover = list.get(i);
			boolean result = handleHandoverAuthor(handover,tenantid, identifier);
			resultMap.put(handover, String.valueOf(result));
		 }
		 return resultMap;
	 }
	
	/**
	 * 接收移交工作
	 * @param handover 工作移交记录表
	 * @param identifier 数据源标识id
	 * @return 接收移交工作是否成功
	 */
	private boolean handleHandoverAuthor(DBHandover handover,String tenantid,String identifier){
		ITaskWork work = TaskWork.getTaskWork();
		work.setUserid(StaticVarExtend.Manager_Log_Userid + Thread.currentThread().getId());
		work.setWorkid(handover.getWorkid());
		work.setTrackid(handover.getTrackid());
		work.setFlowIdentifier(identifier);
		work.setDataIdentifier(identifier);
		work.setTenantid(tenantid);
		work.setClassNameForAction("com.horizon.wf.action.manage.ActionHandoverWork");
		work.setVariable("handover", handover);
		work.setVariable("urlapp", StaticVarExtend.Todo_Default_URL.replace("|workid|", handover.getWorkid())+"&dbIdentifier="+identifier+"&isembedded=false");
		work.executeWork();
		return work.getRunningdata().getResultCode() == StaticVarExtend.F_STATUS_Success;
	}
	
	/**
	 * 激活流程
	 * @param workid 实例id
	 * @param trackid 路径id
	 * @param interfaceClass 激活时要执行的类
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 激活是否成功
	 */
	public boolean activeTrack(String workid,String trackid,String actionname,String interfaceClass,String tenantid,String identifier) throws Exception{

		ITaskWork work = TaskWork.getTaskWork();
		work.setWorkid(workid);
		work.setTrackid(trackid);
		work.setUserid(StaticVarExtend.Manager_Log_Userid + Thread.currentThread().getId());
		work.setClassNameForAction("com.horizon.wf.action.manage.ActionActiveTrack");
		work.setTenantid(tenantid);
		work.setFlowIdentifier(identifier);
		work.setVariable("Active", "1");//1:激活
		work.setVariable("Funname","Active");//操作id
		work.setVariable("interfaceClass",interfaceClass);//执行接口类
		work.setVariable("Actionname", actionname);
		work.setFlowIdentifier(identifier);
		work.executeWork();
		int result = work.getRunningdata().getResultCode() ;
		return result== StaticVarExtend.F_STATUS_Success || result == StaticVarExtend.Init_NoTrack;
	}
	
	/**
	 * 恢复实例数据
	 * @param workid 实例id
	 * @param version 恢复到指定的版本号
	 * @param tenantid 租户id
	 * @return 恢复实例数据是否成功
	 */
	public boolean replaceWork(String workid,int version,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return replaceWork(workid,version,tenantid,identifier);
	}
	/**
	 * 恢复实例数据
	 * @param workid 实例id
	 * @param version 恢复到指定的版本号
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 恢复实例数据是否成功
	 */
	public boolean replaceWork(String workid,int version,String tenantid,String identifier){
		if(StringUtilExtend.isNull(workid) || version == 0){
			return false;
		}
		Map<String,List<Object>> saveMap = new LinkedHashMap<String, List<Object>>();
		IAppInterface[] appInterface = PluginsUtil.getMultiInstance().getAppInterface();
		if(appInterface!=null){
			for (IAppInterface appImpl:appInterface) {
				appImpl.setTenantid(tenantid);
				appImpl.replaceAppData(workid, version, saveMap);//恢复业务的sql
			}
		}
		Map<String,List<Object>> workSqlMap = InstanceManagement.getRevertToTheSpecifiedVersionSQLMap(workid, version, identifier);
		if(null == workSqlMap){
			return false;
		}
		saveMap.putAll(workSqlMap);//恢复实例数据
		PubInfo pubInfo = PubInfo.getPubInfo(tenantid);
		Map<String,String> para = new HashMap<String,String>();
		para.put("workid", workid);
		para.put("trackid",  "");
		para.put("identifier", identifier);
		para.put("actionname",pubInfo.getInfo("Magr_Msg0003", "恢复实例版本"));
		para.put("actionUserid", StaticVarExtend.Manager_Log_Userid);
		para.put("actionUsername", pubInfo.getInfo("Manager_Log_Username","系统管理员"));
		para.put("memo", pubInfo.getInfo("Magr_Msg0004","恢复到|version|版本")
							.replace("|version|", String.valueOf(version)));
		
		DBLog trackLog = StaticFunExtend.getActionCommon().makeDbLogForMgr(para);
		trackLog.getSQL(saveMap);
		
		restoreOperatorRecord(workid,version,saveMap,identifier);//恢复操作过程中的记录数据
		
		PluginsUtil.getSingleInstance().getTaskInterface().restore(workid, version, saveMap, identifier);
		if(!saveMap.isEmpty()){
			RunMsgUtil.getInstance().getStaticInfo().printMapSQL(saveMap);
			if(AccessUtil.getInstance().executeMultiUpdate(saveMap, identifier)){
				MemCacheUtil.removeByWorkid(workid, tenantid);
				return true;
			}
		}
		return false;
	}

	/**
	 * 获取当前实例的最大版本号
	 * @param workid 实例id
	 * @param tenantid 租户id
	 * @return 当前实例的最大版本号
	 */
	public int getWorkMaxVersion(String workid,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return getWorkMaxVersion(workid,tenantid,identifier);
	}
	/**
	 * 获取当前实例的最大版本号
	 * @param workid 实例id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 当前实例的最大版本号
	 */
	public int getWorkMaxVersion(String workid,String tenantid,String identifier){
		if(StringUtilExtend.isNull(workid)  || StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workid+";identifier="+identifier+";");
			return 0;
		}
		return DBWork.getMaxVersionByWorkid(workid, identifier);
	}


	/**
	 * 获取实例存在的版本列表
	 * @return
	 */
	public List<Integer> getInstanceVers(String workid,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return getInstanceVers(workid,tenantid,identifier);
	}
	/**
	 * 获取实例存在的版本列表
	 * @return
	 */
	public List<Integer> getInstanceVers(String workid,String tenantid,String identifier){
		if(StringUtilExtend.isNull(workid)  || StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workid+";identifier="+identifier);
			return null;
		}
		String sql;
		if(HZResourceBundle.getInstance().isVerUseJson()){
			sql = "SELECT VERSION FROM TW_HZ_JSON_VER WHERE WORKID=? ORDER BY VERSION";
		}
		else{
			sql = "SELECT VERSION FROM "+StaticVarExtend.Table_Work_Ver +" WHERE WORKID=? ORDER BY VERSION";
		}
		List<Object> para = new ArrayList<Object>();
		para.add(workid);
		List<List<Object>> result = AccessUtil.getInstance().getMultiList(sql,para,identifier);
		List<Integer> back = new ArrayList<Integer>();
		if(result == null || result.isEmpty()){
			return back;
		}
		for(int i=0,n=result.size();i<n;i++){
			List<Object> tmp = result.get(i);
			back.add(new Integer(""+tmp.get(0)));
		}

		return back;
	}

	/**
	 * 暂停/恢复指定workid的流程实例
	 * @param workid 实例id
	 * @param trackid 对应的路径id
	 * @param tenantid 租户id
	 * @return 暂停/恢复流程实例是否成功
	 */
	public boolean pauseWork(String workid,String trackid,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return pauseWork(workid,trackid,tenantid,identifier);
	}
	/**
	 * 暂停/恢复指定workid的流程实例
	 * @param workid 实例id
	 * @param trackid 对应的路径id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 暂停/恢复流程实例是否成功
	 */
	public boolean pauseWork(String workid,String trackid,String tenantid,String identifier){
		if(StringUtilExtend.isNull(workid) || StringUtilExtend.isNull(trackid) || StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workid+";trackid="+trackid+";identifier="+identifier+";");
			return false;
		}
		return stopOrPauseWork(workid,trackid,tenantid,"com.horizon.wf.action.ActionPause",identifier);
	}
	
	/**
	 * 批量暂停/恢复流程实例
	 * @param workids 实例id数组
	 * @param trackids 对应的路径id数组
	 * @param tenantid 租户id
	 * @return 返回 暂停/恢复流程实例失败的workid,返回空表示全部成功
	 */
	public String pauseWork(String[] workids,String[] trackids,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return pauseWork(workids,trackids,tenantid,identifier);
	}
	/**
	 * 批量暂停/恢复流程实例
	 * @param workids 实例id数组
	 * @param trackids 对应的路径id数组
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 返回 暂停/恢复流程实例失败的workid,返回空表示全部成功
	 */
	public String pauseWork(String[] workids,String[] trackids,String tenantid,String identifier){
		if(CollectionUtil.isStringArrayEmpty(workids) || CollectionUtil.isStringArrayEmpty(trackids) || StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workids+";trackid="+trackids+";identifier="+identifier+";");
			return "";
		}
		StringBuilder sb = new StringBuilder(1000);
		//循环处理
		for(int i=0,n=workids.length;i<n;i++){
			boolean result = pauseWork(workids[i],trackids[i],tenantid,identifier);
			if(!result){
				sb.append(",").append(workids[i]);
			}
//			System.out.println("[暂停流程]workid=" + workids[i]+",结果:"+result);
		}
		return sb.length()>0?sb.deleteCharAt(0).toString():"";
	}
	
	
	/**
	 * 结束实例
	 * @param workid 实例id
	 * @param trackid 对应路径id
	 * @param tenantid 租户id
	 * @return 结束实例是否成功
	 */
	public boolean stopWork(String workid,String trackid,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return stopWork(workid,trackid,tenantid,identifier);
	}
	/**
	 * 结束实例
	 * @param workid 实例id
	 * @param trackid 对应路径id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 结束实例是否成功
	 */
	public boolean stopWork(String workid,String trackid,String tenantid,String identifier){
		if(StringUtilExtend.isNull(workid) || StringUtilExtend.isNull(trackid) || StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workid+";trackid="+trackid+";identifier="+identifier+";");
			return false;
		}
		
		return stopOrPauseWork(workid,trackid,tenantid,"com.horizon.wf.action.ActionStop",identifier);
	}
	/**
	 * 终止流程和暂停流程实际执行
	 * @param workid
	 * @param trackid
	 * @param tenantid
	 * @param className
	 * @param identifier
	 * @return
	 */
	private boolean stopOrPauseWork(String workid,String trackid,String tenantid,String className,String identifier){
		//1.打开实例
		WorkParaBean paraBean = new WorkParaBean();
		paraBean.setFlowIdentifier(identifier);
		paraBean.setUserId(StaticVarExtend.Manager_Log_Userid);
//		paraBean.setUserName(StaticVarExtend.Manager_Log_Username);
		paraBean.setTrackId(trackid);
		paraBean.setWorkId(workid);
		paraBean.setTenantCode(tenantid);
		
		IWorkflowOperator operator = WorkflowFactory.getWorkflowOperator();
		IWorkResult resultBean = operator.open(paraBean);
		if(resultBean.getResult() == StaticVarExtend.Init_Success){
			int status = resultBean.getCurTrackInfo().getFlowstatus();
			if(status == StaticVarExtend.FlowStatus_ErrorEnd 
					|| status == StaticVarExtend.FlowStatus_End){
				if(className.equals("com.horizon.wf.action.ActionStop")){				
					return true;
				}
			}
			//2.执行操作
			paraBean.setActionClass(className);
			paraBean.setSubmitflag("0");
			resultBean = operator.operator(paraBean);
			if(resultBean.getResult() == StaticVarExtend.F_STATUS_Success){
				return true;
			}
		}
		return false;
	}
	
	/**
	 * 批量结束实例(同一事物)
	 * @param workids 实例id数组
	 * @param trackids 对应路径id数组
	 * @param tenantid 租户id
	 * @return 结束实例是否成功
	 */
	public String stopWork(String[] workids,String[] trackids,String tenantid){
		String identifier  = getIdentifierByTenantid(tenantid);
		return stopWork(workids,trackids,tenantid,identifier);
	}
	/**
	 * 批量结束实例(同一事物)
	 * @param workids 实例id数组
	 * @param trackids 对应路径id数组
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 结束实例是否成功
	 */
	public String stopWork(String[] workids,String[] trackids,String tenantid,String identifier){
		if(CollectionUtil.isStringArrayEmpty(workids) 
				|| CollectionUtil.isStringArrayEmpty(trackids) 
				|| StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workids+";trackid="+trackids+";identifier="+identifier+";");
			return "";
		}
		StringBuilder sb = new StringBuilder(1000);
		//循环处理
		for(int i=0,n=workids.length;i<n;i++){
			boolean result = stopWork(workids[i],trackids[i],tenantid,identifier);
			if(!result){
				sb.append(",").append(workids[i]);
			}
//			System.out.println("[结束流程]workid=" + workids[i]+",结果:"+result);
		}
		return sb.length()>0?sb.deleteCharAt(0).toString():"";
	}
	
	/**
	 * 获取当前节点选择参与者的类型  
	 * @param workid 实例id
	 * @param trackid 路径id
	 * @param tenantid 租户id
	 * @return 0:异常 1：单选  2：多选
	 */
	public int getSelectType(String workid,String trackid,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return getSelectType(workid,trackid,tenantid,identifier);
	}
	/**
	 * 获取当前节点选择参与者的类型  
	 * @param workid 实例id
	 * @param trackid 路径id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 0:异常 1：单选  2：多选
	 */
	public int getSelectType(String workid,String trackid,String tenantid,String identifier){
		int selectType = 0;
		IWorkflowOperator iwo = WorkflowFactory.getWorkflowOperator();
		WorkParaBean wpb = new WorkParaBean();
		wpb.setActionClass("com.horizon.wf.action.ActionSubmit");
		wpb.setDataIdentifier(identifier);
		wpb.setFlowIdentifier(identifier);
		wpb.setFunname(StaticVarExtend.OPERATOR_FUNNAME_SUBMIT);
		wpb.setUserId(StaticVarExtend.System_Id);
//		wpb.setUserName(StaticVarExtend.System_Name);
		wpb.setAllUserids(StaticVarExtend.System_Id);
		wpb.setWorkId(workid);
		wpb.setTrackId(trackid);
		IWorkResult wr = iwo.open(wpb);
		if(StaticVarExtend.Init_Success == wr.getResult()){
			int nodetype = wr.getCurFlowNode().getNodetype();
			if(NodeTypeEnum.Node_Multi_Single == nodetype || 
					NodeTypeEnum.Node_Multi == nodetype ||
							NodeTypeEnum.Node_Order == nodetype){
				selectType = 2;
			}else if(NodeTypeEnum.Node_Progressively == nodetype ||
					NodeTypeEnum.Node_Merger == nodetype || 
							NodeTypeEnum.Node_Single == nodetype){
				selectType = 1;
			}
		}
		iwo.close(workid, trackid, StaticVarExtend.System_Id);
		return selectType;
	}
	
	/**
	 * 替换处理人
	 * @param workid 实例id
	 * @param trackid 对应路径id
	 * @param userids 替换后的办理人id
	 * @param tenantid 租户id
	 * @return 替换处理人是否成功
	 */
	public boolean replaceAuthor(String workid,String trackid,String userids,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return replaceAuthor(workid,trackid,userids,tenantid,identifier);
	}
	/**
	 * 替换处理人
	 * @param workid 实例id
	 * @param trackid 对应路径id
	 * @param userids 替换后的办理人id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 替换处理人是否成功
	 */
	public boolean replaceAuthor(String workid,String trackid,String userids,String tenantid,String identifier){
		if(StringUtilExtend.isNull(workid) || StringUtilExtend.isNull(trackid) || StringUtilExtend.isNull(userids) || StringUtilExtend.isNull(identifier)){
			StaticFunExtend.println("parameter is null.workid="+workid+";trackid="+trackid+";userids="+userids+";identifier="+identifier+";");
			return false;
		}
		
		//1.打开实例
		WorkParaBean paraBean = new WorkParaBean();
		paraBean.setFlowIdentifier(identifier);
		paraBean.setUserId(StaticVarExtend.Manager_Log_Userid);
//		paraBean.setUserName(StaticVarExtend.Manager_Log_Username);
		paraBean.setTrackId(trackid);
		paraBean.setWorkId(workid);
		paraBean.setTenantCode(tenantid);
		
		IWorkflowOperator operator = WorkflowFactory.getWorkflowOperator();
		IWorkResult resultBean = operator.open(paraBean);
		if(resultBean.getResult() == StaticVarExtend.Init_Success){
			//2.执行操作
			paraBean.setActionClass("com.horizon.wf.action.ActionReplaceAuthor");
			paraBean.setSubmitflag("0");
			Map<String,String> selectAuthorMap = new HashMap<String,String>();
			selectAuthorMap.put(StaticVarExtend.AUTH_AUTHOR, userids);
			paraBean.setSelectAuthorMap(selectAuthorMap);
			
			resultBean = operator.operator(paraBean);
			if(resultBean.getResult() == StaticVarExtend.F_STATUS_Success){
			    
				return true;
			}
		}
		return false;
	}
	/**
	 * 批量删除流程实例（同一事物）
	 * @param workids 实例id数组
	 * @param tenantid 租户id
	 * @return 删除实例是否成功
	 */
	public boolean deleteWork(String[] workids,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return deleteWork(workids,tenantid,identifier);
	}
	/**
	 * 批量删除流程实例（同一事物）
	 * @param workids 实例id数组
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 删除实例是否成功
	 */
	public boolean deleteWork(String[] workids,String tenantid,String identifier){
		if(CollectionUtil.isStringArrayNotEmpty(workids)){
			List<Object> delPara = new ArrayList<Object>();
			Map<String,List<Object>> saveMap = new LinkedHashMap<String,List<Object>>();
			for(String workid : workids){
				InstanceManagement.getDeleteInstanceSQL(workid, saveMap);//删除实例sql
				List<String> tmpPara = new ArrayList<String>(1);
				tmpPara.add(workid);
				delPara.add(tmpPara);
			}
			//20180312 删除已办权限中的表单信息记录
			saveMap.put("DELETE FROM " + StaticVarExtend.Table_Done + " WHERE WORKID=?", delPara);
			
			IAppInterface[] appInterface = PluginsUtil.getMultiInstance().getAppInterface();
			Map<String,Map<String, List<Object>>>  appMap =null;
			if(appInterface != null){
				for (IAppInterface appImpl:appInterface) {
					appImpl.setTenantid(tenantid);
					appMap = appImpl.deleteAppData(workids,identifier);//删除业务的sql
					if(CollectionUtil.isMapNotEmpty(appMap) && CollectionUtil.isMapNotEmpty(appMap.get(identifier))){
						saveMap.putAll(appMap.get(identifier));
						appMap.remove(identifier);
					}
				}
			}
			deleteOperatorRecord(workids, saveMap);//删除
			if(!saveMap.isEmpty()){
				if(AccessUtil.getInstance().executeMultiUpdate(saveMap, identifier)){
					if(CollectionUtil.isMapNotEmpty(appMap)){
						for(Map.Entry<String, Map<String, List<Object>>> entry : appMap.entrySet()){
							if(!identifier.equals(entry.getKey())){
							    AccessUtil.getInstance().executeMultiUpdate(entry.getValue(), entry.getKey());
							}
						}
					}
					MemCacheUtil.removeByWorkid(workids,tenantid);
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * 删除单个实例
	 * @param workid 实例id
	 * @param tenantid 租户id
	 * @return 删除实例是否成功
	 */
	public boolean deleteWork(String workid,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return deleteWork(workid,tenantid,identifier);
	}
	/**
	 * 删除单个实例
	 * @param workid 实例id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 删除实例是否成功
	 */
	public boolean deleteWork(String workid,String tenantid,String identifier){
		if(StringUtilExtend.isNotNull(workid)){
			return deleteWork(new String[]{workid},tenantid, identifier);
		}
		return false;
	}
		
	/**
	 * 更新指定实例的定义xml
	 * @param workid 实例id
	 * @param tenantid 租户id
	 * @return 更新指定实例的xml是否成功
	 */
	public boolean updateWork(String workid,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return updateWork(workid,tenantid,identifier);
	}
	/**
	 * 更新指定实例的定义xml
	 * @param workid 实例id
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 更新指定实例的xml是否成功
	 */
	public boolean updateWork(String workid,String tenantid,String identifier){
		if(InstanceManagement.updateInstance(workid,tenantid, identifier)){
			return true;
		}
		return false;
	}
	
	/**
	 * 批量更新实例
	 * @param workids 实例id数组
	 * @param tenantid 租户id
	 * @return 批量更新实例是否成功
	 */
	public Map<String, Boolean> updateWork(String[] workids,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return updateWork(workids,tenantid,identifier);
	}
	/**
	 * 批量更新实例
	 * @param workids 实例id数组
	 * @param tenantid 租户id
	 * @param identifier 数据源标识id
	 * @return 批量更新实例是否成功
	 */
	public Map<String, Boolean> updateWork(String[] workids,String tenantid,String identifier){
		Map<String, Boolean> resultMap = new HashMap<String, Boolean>();
		if(CollectionUtil.isStringArrayNotEmpty(workids)){
			for(String s : workids){
				boolean result = updateWork(s,tenantid, identifier);
				resultMap.put(s, result);
			}
		}
		return resultMap;
	}
		
	/**
	 * 获取删除流程操作过程中产生的记录的sql
	 * @param workids 实例id
	 * @param saveMap 存放删除sql语句的map
	 */
	private void deleteOperatorRecord(String[] workids,Map<String,List<Object>> saveMap){
		List<Object> conditionList = new ArrayList<Object>();
		for(String workid : workids){
			conditionList.add(workid);
		}
		String sql = makeQuestionMark(workids, "WORKID");
		String deleteSubflowRecord = "DELETE FROM "+StaticVarExtend.Table_Subflow_Record + " WHERE "+makeQuestionMark(workids,"parentworkid");//子流程
		String deleteSubflowRecordVer = "DELETE FROM "+StaticVarExtend.Table_Subflow_Record_Ver + " WHERE "+makeQuestionMark(workids,"parentworkid");//子流程
		String deleteSubflowRelation = "DELETE FROM "+StaticVarExtend.Table_Relation_Subflow + " WHERE "+makeQuestionMark(workids,"workid");//子流程
		
		String deleteWorkApp = "DELETE FROM "+StaticVarExtend.Table_WorkApp + " WHERE "+sql;//关联表
		String deleteMutual = "DELETE FROM "+StaticVarExtend.Table_Mutual + " WHERE "+sql;	//引擎交互中间表
		String deleteHandover = "DELETE FROM "+StaticVarExtend.Table_Handover + " WHERE "+sql;//
		String deleteHandoverSuccess = "DELETE FROM "+StaticVarExtend.Table_Handover_Log + " WHERE "+sql;//
		String deleteRobotTask = "DELETE FROM "+StaticVarExtend.Table_Robot_Task + " WHERE "+sql;//机器节点临时记录库表
//		String deleteTask = "DELETE FROM "+StaticVarExtend.Table_Task + " WHERE "+sql;//任务表
//		String deleteTaskLog = "DELETE FROM "+StaticVarExtend.Table_Task_Log + " WHERE "+sql;//催办记录
		String deleteTodoSuccess = "DELETE FROM "+StaticVarExtend.Table_Todo_Success + " WHERE "+sql;//待办发送成功数据
		String deleteTodoTemp = "DELETE FROM "+StaticVarExtend.Table_Todo_Temp + " WHERE "+sql;//待办发送临时数据
		String deleteNodeMsg = "DELETE FROM "+StaticVarExtend.Table_Node_Message_Trigger + " WHERE "+sql;//节点消息事件记录
		String deleteNodeMsgLog = "DELETE FROM "+StaticVarExtend.Table_Node_Message_Trigger_Log + " WHERE "+sql;//节点消息事件日志记录
		String deleteNodeSignal = "DELETE FROM "+StaticVarExtend.Table_Node_Signal_Trigger + " WHERE "+sql;//节点信号记录
		String deleteNodeSignalLog = "DELETE FROM "+StaticVarExtend.Table_Node_Signal_Trigger_Log + " WHERE "+sql;//节点信号记录
		String deleteNodeTask = "DELETE FROM "+StaticVarExtend.Table_Node_Task_Trigger + " WHERE "+sql;//节点任务记录
		String deleteNodeTaskLog = "DELETE FROM "+StaticVarExtend.Table_Node_Task_Trigger_log + " WHERE "+sql;//节点任务记录
		String deleteRelationTimmer = "DELETE FROM "+StaticVarExtend.Table_Relation_Timer + " WHERE "+sql;//定时任务和任务记录关联表
		String deleteRelationTimmerLog = "DELETE FROM "+StaticVarExtend.Table_Relation_Timer_Log + " WHERE "+sql;//定时任务和任务记录关联表
//		String deleteFlowVar = "DELETE FROM "+StaticVarExtend.Table_Flowvar_Record + " WHERE "+sql;//流程变量
		
		saveMap.put(deleteSubflowRecord, conditionList);
		saveMap.put(deleteSubflowRecordVer, conditionList);
		saveMap.put(deleteSubflowRelation, conditionList);
		
		saveMap.put(deleteWorkApp, conditionList);
		saveMap.put(deleteMutual, conditionList);
		saveMap.put(deleteHandover, conditionList);
		saveMap.put(deleteHandoverSuccess, conditionList);
		saveMap.put(deleteRobotTask, conditionList);
//		saveMap.put(deleteTask, conditionList);
//		saveMap.put(deleteTaskLog, conditionList);
		saveMap.put(deleteTodoSuccess, conditionList);
		saveMap.put(deleteTodoTemp, conditionList);
		saveMap.put(deleteNodeMsg, conditionList);
		saveMap.put(deleteNodeMsgLog, conditionList);
		saveMap.put(deleteNodeSignal, conditionList);
		saveMap.put(deleteNodeSignalLog, conditionList);
		saveMap.put(deleteNodeTask, conditionList);
		saveMap.put(deleteNodeTaskLog, conditionList);
		saveMap.put(deleteRelationTimmer, conditionList);
		saveMap.put(deleteRelationTimmerLog, conditionList);
//		saveMap.put(deleteFlowVar, conditionList);
	}
	
	/**
	 * 根据给定的workids生成sql中的文号和参数
	 * @param para sql语句
	 * @return 拼装好的sql语句
	 */
	private String makeQuestionMark(String[] workids,String para){
		StringBuilder qm = new StringBuilder("");
		int len = workids.length;
		if(len == 1){
			qm.append(" ");
			qm.append(para);
			qm.append(" =?");
		}else{
			qm.append(" ");
			qm.append(para);
			qm.append(" IN(");
			for(int i = 1;i<=len;i++){
				qm.append("?");
				if(len != i){
					qm.append(",");
				}
			}
			qm.append(")");
		}
		return qm.toString();
	}

	/**
	 * 恢复操作数据
	 * @param workid 实例id
	 * @param version 版本号
	 */
	private void restoreOperatorRecord(String workid, int version,Map<String,List<Object>> saveMap,String identifier) {
		/*
		List<DBHuiqianRecord> records = getHuiqianRecordVerByWorkid(workid, version, identifier);
		if(CollectionUtil.isListNotEmpty(records)){
			String deleteSql = "DELETE FROM "+StaticVarExtend.Table_Huiqian_Record + " WHERE WORKID=? AND AUTH_VERSION>?";
			String deleteSqlVer = "DELETE FROM "+StaticVarExtend.Table_Huiqian_Record_Ver + " WHERE WORKID=? AND AUTH_VERSION>=?";
			List<Object> conditionList = new ArrayList<Object>();
			conditionList.add(workid);
			conditionList.add(version);//20140930liys增加
			saveMap.put(deleteSql, conditionList);
			saveMap.put(deleteSqlVer, conditionList);
			
			for(int i=0,n=records.size();i<n;i++){
				DBHuiqianRecord r = records.get(i);
				r.setId(r.getRecordid());
				r.getSQL(saveMap);
			}
		}
		//*/
		List<Object> cList = new ArrayList<Object>(2);
		cList.add(workid);
		cList.add(version);
//		这个记录数据咋恢复呢???
		String delSubRecordSql = "DELETE FROM "+StaticVarExtend.Table_Subflow_Record +" WHERE PARENTWORKID =? AND VERSION>=?";
		String delSubRecordVerSql = "DELETE FROM "+StaticVarExtend.Table_Subflow_Record_Ver +" WHERE PARENTWORKID =? AND VERSION>?";
		saveMap.put(delSubRecordSql, cList);
		saveMap.put(delSubRecordVerSql, cList);
		
		String delSubRelationSql = "DELETE FROM "+StaticVarExtend.Table_Relation_Subflow +" WHERE WORKID =? AND VERSION>?";
		saveMap.put(delSubRelationSql, cList);
		
		String msgSql = "SELECT * FROM " + StaticVarExtend.Table_Node_Message_Trigger_Log +" WHERE WORKID=? AND VERSION=?";
		
		List<DBNodeMessageTrigger> msgTriggers = AccessUtil.getInstance().getMultiObject(msgSql, cList, DBNodeMessageTrigger.class, identifier);
		if(CollectionUtil.isListNotEmpty(msgTriggers)){
			for(int i=0,n=msgTriggers.size();i<n;i++){
				DBNodeMessageTrigger msg = msgTriggers.get(i);
				msg.setStatus("1");
				msg.getSQL(saveMap,null);
			}
		}
		
		String taskSql = "SELECT * FROM " + StaticVarExtend.Table_Node_Task_Trigger_log +" WHERE WORKID=? AND VERSION=?";
		List<DBNodeTaskTrigger> taskTriggers = AccessUtil.getInstance().getMultiObject(taskSql, cList, DBNodeTaskTrigger.class, identifier);
		if(CollectionUtil.isListNotEmpty(taskTriggers)){
			for(int i=0,n=taskTriggers.size();i<n;i++){
				DBNodeTaskTrigger msg = taskTriggers.get(i);	
				msg.setStatus("1");
				msg.getSQL(saveMap,null);
			}
		}

		String signalSql = "SELECT * FROM " + StaticVarExtend.Table_Node_Signal_Trigger_Log +" WHERE WORKID=? AND VERSION=?";
		List<DBNodeSignalTrigger> signalTriggers = AccessUtil.getInstance().getMultiObject(signalSql, cList, DBNodeSignalTrigger.class, identifier);
		if(CollectionUtil.isListNotEmpty(signalTriggers)){
			for(int i=0,n=signalTriggers.size();i<n;i++){
				DBNodeSignalTrigger msg = signalTriggers.get(i);
				msg.setStatus("1");
				msg.getSQL(saveMap,null);
			}
		}
		
		
		
	}
	/**
	 * 查询实例发起会签的记录数(管理接口调用)
	 * @param workid 实例id
	 * @param version 版本
	 * @param identifier 数据库标识符
	 * @return 会签记录
	 */
	/*
	private List<DBHuiqianRecord> getHuiqianRecordVerByWorkid(String workid,int version, String identifier){
		String sql = "SELECT * FROM "
				+ StaticVarExtend.Table_Huiqian_Record_Ver + " WHERE WORKID=? AND OLD_VERSION=?";
		List<Object> conditionList = new ArrayList<Object>();
		conditionList.add(workid);
		conditionList.add(version);
		return AccessUtil.getInstance().getMultiObject(sql, conditionList, DBHuiqianRecord.class, identifier);
	}
	//*/
	
//===============================================================================	
	private String getTaskSql = "SELECT * FROM "+StaticVarExtend.Table_Task+" A WHERE 1=1  ";
	public List<DBTask> getTask(String where,List<? extends Object> para,String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		return getTask(where,para,tenantid,identifier);
	}
	public List<DBTask> getTask(String where,List<? extends Object> para,String tenantid,String identifier){
		String sql = getTaskSql;
		if(where == null || "".equals(where)){
			para = null;
		}
		else{
			sql = getTaskSql + where;
		}
		sql = sql + " ORDER BY A.FLOWNAME,A.STARTTIME DESC";
		return AccessUtil.getInstance().getMultiObject(sql, para, DBTask.class, identifier);
	}
	
	/**
	 * 代办设置
	 * @param hashMap
	 * @return
	 */
	public boolean agentUserSet(Map<String,String> hashMap, String tenantid) {
		String identifier = getIdentifierByTenantid(tenantid);
		return agentUserSet(hashMap,tenantid,identifier);
	}
	public boolean agentUserSet(Map<String,String> hashMap,String tenantid, String identifier) {
		AgentUtil agentUtil = AgentUtil.getInstance();
		boolean result = agentUtil.agentUserSet(hashMap, tenantid, identifier);
		return result;
	}
//===============================================================================	
//以下为测试代码	
//===============================================================================		
	/**
	 * 清除所有实例相关表中以TW开头的表数据,tw_hz_flow_def除外
	 */
	public void clearAllInstance(String tenantid){
		String identifier = getIdentifierByTenantid(tenantid);
		clearAllInstance(tenantid,identifier);
	}
	public void clearAllInstance(String tenantid,String identifier){
		Map<String,List<Object>> saveMap = new LinkedHashMap<String,List<Object>>();
		List<Object> lst = AccessUtil.getInstance().getAllTablesNameList(identifier);
		String sql = "DELETE FROM table";
		for(int i=0,n=lst.size();i<n;i++){
			String tblName = (String)lst.get(i);
			if(tblName.startsWith("TW") && 
					(!("tw_hz_flow_def".equalsIgnoreCase(tblName)
					   ||"tw_hz_flow_auth".equalsIgnoreCase(tblName)
					   ||"tw_hz_form_field".equalsIgnoreCase(tblName)
					   ||"tw_hz_form_fieldauth".equalsIgnoreCase(tblName))
					)){
				saveMap.put(sql.replace("table", tblName), null);
			}
		}
	
		AccessUtil.getInstance().executeMultiUpdate(saveMap, identifier);
	}

	/**
	 * 把实例从结束库表中恢复到运行库中
	 * @param workid  实例id
	 * @param suffix 实例所在库表的后缀（相对运行库表名称时添加的后缀比如默认的[_END]）
	 * @param identifier 数据源标识符
	 * @return
	 */
	public boolean moveEndInstanceToRunning(String workid,String suffix,String identifier){
		Map<String,List<Object>> saveMap = new LinkedHashMap<String,List<Object>>();
		//获取需要恢复的数据SQL
		InstanceManagement.getCopyEndToRunningSQL(workid,suffix,saveMap);

		if(!saveMap.isEmpty()){
			if(AccessUtil.getInstance().executeMultiUpdate(saveMap, identifier)){
				/*  如需要同时处理其他数据，可以在此添加处理
				if(CollectionUtil.isMapNotEmpty(appMap)){
					for(Map.Entry<String, Map<String, List<Object>>> entry : appMap.entrySet()){
						if(!identifier.equals(entry.getKey())){
							AccessUtil.getInstance().executeMultiUpdate(entry.getValue(), entry.getKey());
						}
					}
				}
				//*/
				MemCacheUtil.removeByWorkid(workid);
				return true;
			}
		}
		return false;
	}

	/**
	 * 转移已经结束的实例到其他数据表中(不包含业务数据) 
	 * @param workid  实例id
	 * @param suffix  要转移的库表的后缀（相对运行库表名称时添加的后缀比如默认的[_END]）
	 * @param identifier 数据源标识符
	 */
	public boolean moveInstanceToOtherIdentifier(String workid,String suffix,String identifier){
		Map<String,List<Object>> saveMap = new LinkedHashMap<String,List<Object>>();
		//获取需要恢复的数据SQL
		InstanceManagement.getCopyRunningToEndSQL(workid,suffix,saveMap);

		if(!saveMap.isEmpty()){
			if(AccessUtil.getInstance().executeMultiUpdate(saveMap, identifier)){
				/*  如需要同时处理其他数据，可以在此添加处理
				if(CollectionUtil.isMapNotEmpty(appMap)){
					for(Map.Entry<String, Map<String, List<Object>>> entry : appMap.entrySet()){
						if(!identifier.equals(entry.getKey())){
							AccessUtil.getInstance().executeMultiUpdate(entry.getValue(), entry.getKey());
						}
					}
				}
				//*/
				MemCacheUtil.removeByWorkid(workid);
				return true;
			}
		}
		return false;
	}

	/**
	 * 为实例添加仅可查看权限
	 * @param workId 实例id
	 * @param users 目标用户 U_userid 用户 || D_deptid 部门 || P_posid 岗位 || G_groutid 群组
	 * @param tenantId 租户标识
	 * @param identifier 数据源标识
	 * @return
	 */
	public String addCReader(String workId,List<String> users,String tenantId,String identifier){
		String result=FlowDBUtil.addCReader(workId,users,identifier);
		if("success".equals(result)){
			MemCacheUtil.removeByWorkid(workId,tenantId);
		}
		return result;
	}

}
