package com.horizon.workflow.support.init;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.horizon.common.base.Objects;
import com.horizon.common.base.Predicate;
import com.horizon.common.base.Splitter;
import com.horizon.common.collect.ArrayListMultimap;
import com.horizon.common.collect.BiMap;
import com.horizon.common.collect.Collections2;
import com.horizon.common.collect.HashBiMap;
import com.horizon.common.collect.ListMultimap;
import com.horizon.common.collect.Maps;
import com.horizon.core.init.HInit;
import com.horizon.core.init.HorizonInitAble;
import com.horizon.exception.extend.InitException;
import com.horizon.workflow.support.action.ActionBaseStrategy;

import static com.horizon.wf.global.StaticVarExtend.*;

/**
 * 初始化流程运行框架参数
 *
 * @author mwr
 */
@HInit
public class WorkflowOperateInit implements HorizonInitAble {

    private  static final Logger LOGGER = LoggerFactory.getLogger(WorkflowOperateInit.class);

    /**
     * 流程结果处理策略
     */
    protected  static final Map<String, String> strategy = Maps.newHashMap();

    /**
     * 操作类型
     */
    protected  static final ListMultimap<String, String> action = ArrayListMultimap.create();

    /**
     * 办理人
     */
    protected  static final BiMap<String, String> author = HashBiMap.create();

    @Override
    public void start() {

        LOGGER.info(" 初始化流程运行框架模块");
        initAction();
        initActionType();
        initAuthor();
    }

    /**
     * 初始化操作结果的策略
     */
    protected final void initAction() {

        strategy.put("submit", "com.horizon.workflow.support.action.DefaultSubmitAction");
        strategy.put("freeSubmit", "com.horizon.workflow.support.action.FreeSubmitAction");
        strategy.put("author", "com.horizon.workflow.support.action.AuthorSubmitAction");
        strategy.put("single", "com.horizon.workflow.support.action.SingleSubmitAction");
        strategy.put("save", "com.horizon.workflow.support.action.SingleSaveSubmitAction");
        strategy.put("claim", "com.horizon.workflow.support.action.SingleClaimSubmitAction");
        strategy.put("reduce", "com.horizon.workflow.support.action.ReduceSubmitAction");
        strategy.put("reader", "com.horizon.workflow.support.action.ReaderSubmitAction");
        strategy.put("subflow", "com.horizon.workflow.support.action.SubflowAuthorsSubmitAction");
        strategy.put("inner", "com.horizon.workflow.support.action.InnerSubmitAction");
        strategy.put("reject", "com.horizon.workflow.support.action.RejectSubmitAction");
        strategy.put("jumpreject", "com.horizon.workflow.support.action.RejectJumpSubmitAction");

        WorkflowOperateFactory operateFactory = WorkflowOperateFactory.newInstance();
        if (operateFactory != null) {
            Map<String, String> strategyMap = operateFactory.getStrategy();
            if (strategyMap != null && !strategyMap.isEmpty()) {
                for (Entry<String, String> entry : strategyMap.entrySet()) {
                    strategy.put(entry.getKey(), entry.getValue());
                }
            }
        }
    }

    /**
     * 初始化操作类型分类
     */
    protected final void initActionType() {

        action.put("submit", OPERATOR_FUNNAME_SUBMIT);
        action.put("submit", OPERATOR_FUNNAME_STARTFLOW);
        action.put("submit", OPERATOR_FUNNAME_GOTO);
        action.put("submit", OPERATOR_FUNNAME_REISSUE_ROUTER);
        action.put("freeSubmit", OPERATOR_FUNNAME_FREE_SUBMIT);

        action.put("jumpreject", OPERATOR_FUNNAME_JUMP_REJECT);
        action.put("inner", OPERATOR_FUNNAME_INNER_SUBMIT);
        action.put("reject", OPERATOR_FUNNAME_REJECT);
        action.put("subflow", OPERATOR_FUNNAME_SUBFLOW);
        action.put("reduce", OPERATOR_FUNNAME_REDUCE_AUTHOR);
        action.put("reader", OPERATOR_FUNNAME_ADD_READER);
        action.put("claim", OPERATOR_FUNNAME_CLAIM);
        action.put("save", OPERATOR_FUNNAME_SAVE);

        action.put("author", OPERATOR_FUNNAME_REPLACE_AUTHOR);
        action.put("author", OPERATOR_FUNNAME_MOVE);
        action.put("author", OPERATOR_FUNNAME_REISSUE);
        action.put("author", OPERATOR_FUNNAME_JIAQIAN);
        action.put("author", OPERATOR_FUNNAME_JIAQIAN_BEFORE);
        action.put("author", OPERATOR_FUNNAME_HUIQIAN);
        action.put("author", OPERATOR_FUNNAME_ADD_SECONDAUTHOR);

        action.put("single", OPERATOR_FUNNAME_READED);
        action.put("single", OPERATOR_FUNNAME_REJECT_SUBMIT);
        action.put("single", OPERATOR_FUNNAME_INNER_REJECT);
        action.put("single", OPERATOR_FUNNAME_STOP);
        action.put("single", OPERATOR_FUNNAME_PAUSE);
        action.put("single", OPERATOR_FUNNAME_RESTORE);
        action.put("single", OPERATOR_FUNNAME_GETBACK);
        action.put("single", OPERATOR_FUNNAME_GRAB);
        action.put("single", OPERATOR_FUNNAME_CANCEL);
        action.put("single", OPERATOR_FUNNAME_HUIQIAN_SUBMIT);
        action.put("single", OPERATOR_FUNNAME_SENCOND_SUBMIT);
        action.put("single", OPERATOR_FUNNAME_URGE);

        WorkflowOperateFactory operateFactory = WorkflowOperateFactory.newInstance();
        if (operateFactory != null) {
            Map<String, String> actionMap = operateFactory.getAction();
            if (actionMap != null && !actionMap.isEmpty()) {
                for (Entry<String, String> entry : actionMap.entrySet()) {
                    String value = entry.getValue();
                    List<String> list = Splitter.on(";").omitEmptyStrings().splitToList(value);
                    action.putAll(entry.getKey(), list);
                }
            }
        }
    }

    /**
     * 初始化操作人名称
     */
    protected final void initAuthor() {

        author.put(AUTH_AUTHOR, "主办");
        author.put(AUTH_SECOND_AUTHOR, "协办");
        author.put(AUTH_SUBFLOW_AUTHOR, "子流程发起人");
        author.put(AUTH_START_HUIQIAN_AUTHOR, "会签发起人");
        author.put(AUTH_HUIQIAN_AUTHOR, "会签办理人");
        author.put(AUTH_READER, "读者");
        author.put(AUTH_READED, "已阅(读者)");
        author.put(AUTH_CREADER, "读者*");
        author.put(AUTH_DONE, "已办");
        author.put(AUTH_DONE_HUIQIAN, "已办(会签)");
        author.put(AUTH_DONE_SecondSubmit, "已办(协办)");
        author.put(AUTH_CREATOR, "创建者");
        author.put(AUTH_MANAGER, "管理员");

        WorkflowOperateFactory operateFactory = WorkflowOperateFactory.newInstance();
        if (operateFactory != null) {
            Map<String, String> aurhorMap = operateFactory.getAuthor();
            if (aurhorMap != null && !aurhorMap.isEmpty()) {
                for (Entry<String, String> entry : aurhorMap.entrySet()) {
                    author.put(entry.getKey(), entry.getValue());
                }
            }
        }

    }

    @Override
    public void destory() {

        strategy.clear();
        action.clear();
        author.clear();

    }

    @Override
    public int getSort() {

        return 12;
    }

    /**
     * 根据办理人ID获取办理人名称
     *
     * @param authorId
     *            办理人ID
     * @return
     */
    public static final String getAuhorName(String authorId) {

        return author.get(authorId);
    }

    /**
     * 根据办理人名称获取办理人ID
     *
     * @param authorName
     *            办理人名称
     * @return
     */
    public static final String getAuhorId(String authorName) {

        return author.inverse().get(authorName);
    }

    /**
     * 根据操作返回结果处理策略类
     *
     * @param operate
     * @return
     */
    public static final ActionBaseStrategy getActionBaseStrategy(final String operate) {

        String actionType = "submit";
        Collection<Entry<String, String>> entries = action.entries();
        Collection<Entry<String, String>> filterList = Collections2.filter(entries,
                new Predicate<Entry<String, String>>() {

                    @Override
                    public boolean apply(Entry<String, String> entry) {

                        return Objects.equal(operate.toLowerCase(), entry.getValue().toLowerCase());
                    }
                });
        if (!filterList.isEmpty()) {
            for (Entry<String, String> entry : filterList) {
                actionType = entry.getKey();
            }
        }
        String clazz = strategy.get(actionType);
        return initClass(clazz);
    }

    @SuppressWarnings("unchecked")
    private static <T> T initClass(String clazzName) throws InitException {

        try {
            ClassLoader loader = Thread.currentThread().getContextClassLoader();
            Class<?> clazz = loader.loadClass(clazzName);
            return (T) clazz.newInstance();
        } catch (ClassNotFoundException e) {
            LOGGER.error("找不到加载的类," + e.getMessage());
            throw new InitException("找不到加载的类," + e.getMessage(), e);
        } catch (InstantiationException e) {
            LOGGER.error("该类无法实例化," + e.getMessage());
            throw new InitException("该类无法实例化," + e.getMessage(), e);
        } catch (IllegalAccessException e) {
            LOGGER.error("该类无法实例化,可能缺少空参构造函数," + e.getMessage());
            throw new InitException("该类无法实例化,可能缺少空参构造函数," + e.getMessage(), e);
        }
    }
}
