package com.horizon.third.service;

import com.horizon.common.base.Objects;
import com.horizon.core.HorizonCore;
import com.horizon.core.tree.TreeModel;
import com.horizon.third.OrgConstant;
import com.horizon.third.build.OrgDataBuilder;
import com.horizon.third.build.OrgDataQuery;
import com.horizon.third.build.OrgDirector;
import com.horizon.third.entity.OrgData;
import com.horizon.third.entity.User;
import com.horizon.utils.StringUtil;

import java.util.*;

/**
 * @author pengys 2021/10/13.
 */
public final class OrgDataQueryBuilder implements OrgDataBuilder, OrgConstant {


    private OrgDataQuery orgDataQuery;

    public OrgDataQueryBuilder(OrgDataQuery orgDataQuery){
        this.orgDataQuery = orgDataQuery;
    }

    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @return
     */
    public List<User> findUserByOrgId(String type, String orgId) {
        List<User> users = null;
        String tenantCode = this.getTenantCode();
        if (Objects.equal(DEPT, type)) {
            users= orgDataQuery.findUserByDeptId(tenantCode,orgId);
        } else if (Objects.equal(GROUP, type)) {
            users = orgDataQuery.findUserByGroupId(tenantCode,orgId);
        } else if (Objects.equal(POSITION, type)) {
            users = orgDataQuery.findUserByPositionId(tenantCode,orgId);
        } else if (Objects.equal(ROLE, type)) {
            users = orgDataQuery.findUserByRoleId(tenantCode,orgId);
        }
        if (users == null){
            users = new ArrayList<User>();
        }
        return users;
    }
    /**
     * 根据组织机构的类型和id查找对应给组织机构数据
     * @param type 组织机构类型
     * @param orgId 组织机构ID
     * @return
     */
    public OrgData getOrgDataById(String type, String orgId) {
        OrgData orgData = null;
        String tenantCode = this.getTenantCode();
        if (Objects.equal(DEPT, type)) {
            orgData= orgDataQuery.getDeptById(tenantCode,orgId);
        } else if (Objects.equal(GROUP, type)) {
            orgData = orgDataQuery.getGroupById(tenantCode,orgId);
        } else if (Objects.equal(POSITION, type)) {
            orgData = orgDataQuery.getPositionById(tenantCode,orgId);
        } else if (Objects.equal(ROLE, type)) {
            orgData = orgDataQuery.getRoleById(tenantCode,orgId);
        }
        return orgData;
    }
    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @return
     */
    public List<OrgData> findChildrenData(String type, String orgId) {
        String tenantCode = this.getTenantCode();
        List<OrgData> childrenData = new ArrayList<OrgData>();
        if (Objects.equal(DEPT, type)) {
            childrenData= orgDataQuery.findDeptByParentId(tenantCode,orgId);
        } else if (Objects.equal(GROUP, type)) {
            childrenData = orgDataQuery.findGroupByParentId(tenantCode,orgId);
        } else if (Objects.equal(POSITION, type)) {
            childrenData = orgDataQuery.findPositionByParentId(tenantCode,orgId);
        } else if (Objects.equal(ROLE, type)) {
            childrenData = orgDataQuery.findRoleByParentId(tenantCode,orgId);
        }
        return childrenData;
    }

    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     *@param isAllChild
     * @return
     */
    public List<OrgData> findChildrenData(String type, String orgId, boolean isAllChild) {
        if (!isAllChild){
            return this.findChildrenData(type, orgId);
        }
        List<OrgData> orgDataList = this.findOrgData(type);
        Map<String, List<OrgData>> orgMap = new HashMap<String, List<OrgData>>();
        List<OrgData> childrenData = new ArrayList<OrgData>();
        if (!orgDataList.isEmpty()){
            //orgId是根节点id时
            OrgData rootNode = getRootNode(type);
            if (Objects.equal(rootNode.getId(),orgId)){
                return orgDataList;
            }
            for (OrgData data : orgDataList) {
                String parentId = data.getParentId();
                if (StringUtil.hasValue(parentId)) {
                    if (orgMap.containsKey(parentId)){
                        List<OrgData> childData = orgMap .get(parentId);
                        childData.add(data);
                    }else {
                        List<OrgData> childData = new ArrayList<OrgData>();
                        childData.add(data);
                        orgMap.put(parentId, childData);
                    }
                }
            }
            OrgData orgData = this.getOrgDataById(type, orgId);
            childrenData.add(orgData);
            this.findChildrenData(orgId,childrenData,orgMap);
        }
        return childrenData;
    }
    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @return
     */
    public OrgData getRootNode(String type) {
        OrgData rootNode = null;
        String tenantCode = this.getTenantCode();
        if (Objects.equal(DEPT, type)) {
            rootNode= orgDataQuery.getDeptRoot(tenantCode);
        } else if (Objects.equal(GROUP, type)) {
            rootNode = orgDataQuery.getGroupRoot(tenantCode);
        } else if (Objects.equal(POSITION, type)) {
            rootNode = orgDataQuery.getPositionRoot(tenantCode);
        } else if (Objects.equal(ROLE, type)) {
            rootNode = orgDataQuery.getRoleRoot(tenantCode);
        }
        return rootNode;
    }

    /**
     * 根据传参值模糊检索所有用户
     *
     * @param param
     * @return 单个对象
     */
    public  List<User> getUserLikeParam(String param) {
        return orgDataQuery.findUserLikeParam(this.getTenantCode(),param);
    }

    /**
     * 根据传参值模糊检索用户
     * </P> 根据searchMax属性只搜索最多searchMax值个用户,searchMax<=0时为无效值,无效值会返回所有
     * @param param
     * @return 单个对象
     */
    @Override
    public List<User> getUserLikeParam(String param, int searchMax) {
        return orgDataQuery.findUserLikeParam(this.getTenantCode(), param, searchMax);
    }

    public User getUserById(String id) {
        List<User> users = OrgDirector.USER.getVaules();
        User user = null;
        if (users == null){
            user = orgDataQuery.getUserById(this.getTenantCode(),id);
        }else {
            int index = users.indexOf(new User(id));
            if (index >-1) {
                user = users.get(index);
            }else {
                user = orgDataQuery.getUserById(this.getTenantCode(),id);
            }
        }
        return user;
    }

    /**
     * 查询所有用户
     * @return
     */
    @Override
    public List<User> findAllUser() {
        List<User> users = OrgDirector.USER.getVaules();
        if (users == null ){
            users  = orgDataQuery.findAllUser(this.getTenantCode());
        }
        return users;
    }

    /**
     * 根据登录名检索用户
     *
     * @param loginName
     * @return 单个对象
     */
    public User getUserByLoginName(String loginName) {
        return orgDataQuery.getUseByLoginName(this.getTenantCode(),loginName);
    }

    /**
     * 根据用户id查找用户所在的组织机构
     * @param userId
     * @return
     */
    public List<OrgData> getOrgForUserByUserId(String userId) {
        List<OrgData> results = new ArrayList<OrgData>();
        String tenantCode = this.getTenantCode();
        List<OrgData> depts = orgDataQuery.findDeptByUserId(tenantCode,userId);
        if (depts != null && !depts.isEmpty()){
            results.addAll(depts);
        }
        List<OrgData> groups = orgDataQuery.findGroupByUserId(tenantCode,userId);
        if (groups != null && !groups.isEmpty()){
            results.addAll(groups);
        }
        List<OrgData> positons = orgDataQuery.findPositionByUserId(tenantCode,userId);
        if (positons != null && !positons.isEmpty()){
            results.addAll(positons);
        }
        List<OrgData> roles = orgDataQuery.findRoleByUserId(tenantCode,userId);
        if (roles != null && !roles.isEmpty()){
            results.addAll(roles);
        }
        return results;
    }

    /**
     * 根据用户id和组织机构类型查找用户所在的组织机构
     * @param type
     * @param userId
     * @return
     */
    public List<OrgData> getOrgForUserByUserId(String type, String userId) {
        List<OrgData> results = new ArrayList<OrgData>();
        String tenantCode = this.getTenantCode();
        if (Objects.equal(DEPT, type)) {
            results =  orgDataQuery.findDeptByUserId(tenantCode,userId);
        } else if (Objects.equal(GROUP, type)) {
            results =  orgDataQuery.findGroupByUserId(tenantCode,userId);
        } else if (Objects.equal(POSITION, type)) {
            results =  orgDataQuery.findPositionByUserId(tenantCode,userId);
        } else if (Objects.equal(ROLE, type)) {
            results =  orgDataQuery.findRoleByUserId(tenantCode,userId);
        }
       return results;
    }

    /**
     * 根据组织机构的类型和id查找所有的上级
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @return
     */
    @Override
    public List<OrgData> findParentToRoot(String type, String orgId) {
        List<OrgData> orgDataList = this.findOrgData(type);
        List<OrgData> parentList = new ArrayList<OrgData>();
        if (!orgDataList.isEmpty()){
            Map<String, OrgData> orgMap = new HashMap<String,OrgData>();
            for (OrgData data : orgDataList) {
                String id = data.getId();
                orgMap.put(id,data);
            }
            OrgData orgData = orgMap.get(orgId);
            String parentId = orgData.getParentId();
            this.findParentToRoot(parentId,parentList,orgMap);
        }
        return parentList;
    }
    /**
     * 根据组织机构的类型查找对应的组织机构数据
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @return
     */
    @Override
    public List<OrgData> findOrgData(String type) {
        List<OrgData> results = null;
        if (Objects.equal(DEPT, type)) {
            results = OrgDirector.DEPT.base();
        } else if (Objects.equal(GROUP, type)) {
            results = OrgDirector.GROUP.base();
        } else if (Objects.equal(POSITION, type)) {
            results = OrgDirector.POSITION.base();
        } else if (Objects.equal(ROLE, type)) {
            results = OrgDirector.ROLE.base();
        }
        if ( results == null){
            return Collections.emptyList();
        }
        return results;
    }
    /**
     * 获取所有组织机构类型
     * @param
     * @return
     */
    @Override
    public List<OrgData> findOrgData() {
        List<OrgData> results = new ArrayList<OrgData>();
        List<OrgData> depts = this.findOrgData(DEPT);
        if (!depts.isEmpty()){
            results.addAll(depts);
        }
        List<OrgData> groups = this.findOrgData(GROUP);
        if (!groups.isEmpty()){
            results.addAll(groups);
        }
        List<OrgData> positions = this.findOrgData(POSITION);
        if (!positions.isEmpty()){
            results.addAll(positions);
        }
        List<OrgData> roles = this.findOrgData(ROLE);
        if (!roles.isEmpty()){
            results.addAll(roles);
        }
        return results;
    }

    /**
     * 根据类型获得内存树
     *
     * @param type
     *            D:部门书；G:群组；P:岗位；R:角色
     * @return
     */
    @Override
    public TreeModel<OrgData> getOrgTree(String type) {
        List<OrgData> orgData = this.findOrgData(type);
        return this.buildTree(orgData);
    }

    /**
     * 递归查找所有的上级
     * @param orgId
     * @param orgDataList
     * @param orgMap
     */
    private void findParentToRoot(String orgId,List<OrgData> orgDataList,Map<String, OrgData> orgMap) {
        OrgData parent = orgMap.get(orgId);
        if (parent != null){
            orgDataList.add(parent);
            String parentId = parent.getParentId();
            if (StringUtil.hasValue(parentId)){
                findParentToRoot(parentId,orgDataList,orgMap);
            }

        }
    }
    /**
     * 递归查找所有的下级
     * @param orgId
     * @param childrenData
     * @param orgMap
     */
    public void findChildrenData(String orgId, List<OrgData> childrenData, Map<String, List<OrgData>> orgMap) {
        List<OrgData> children = orgMap.get(orgId);
        if (children != null && !children.isEmpty()){
            childrenData.addAll(children);
            for (OrgData child : children) {
                String id = child.getId();
                findChildrenData(id,childrenData,orgMap);
            }
        }
    }
    /**
     * 构建树
     *
     * @param datas
     * @return
     */
    private TreeModel<OrgData> buildTree(List<OrgData> datas) {

        TreeModel<OrgData> tree = null;
        if (datas != null) {
            tree = new TreeModel<OrgData>().creat(datas);
        }
        return tree;
    }

    /**
     * 构建树
     *
     * @return
     */
    private String getTenantCode() {
        return HorizonCore.TENANT_CODE.value();
    }

}
