package com.horizon.third.service;

import com.horizon.common.base.Objects;
import com.horizon.common.collect.Lists;
import com.horizon.core.tree.TreeModel;
import com.horizon.core.tree.TreeNode;
import com.horizon.third.OrgConstant;
import com.horizon.third.build.OrgDataBuilder;
import com.horizon.third.build.OrgDirector;
import com.horizon.third.entity.OrgData;
import com.horizon.third.entity.User;
import com.horizon.utils.StringUtil;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author pengys 2021/10/14.
 */
public final class OrgDataDefaultBuilder implements OrgDataBuilder, OrgConstant {

    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @return
     */
    @Override
    public List<User> findUserByOrgId(String type, String orgId) {
        TreeModel<OrgData> treeModel = this.getOrgTree(type);
        if (treeModel != null) {
            TreeNode<OrgData> node = treeModel.getNode(orgId);
            if (node != null){
                OrgData orgData = node.getData();
                if (orgData != null) {
                    List<User> users = orgData.getUsers();
                    return users;
                }
            }
        }
        return Collections.emptyList();
    }
    /**
     * 根据组织机构的类型和id查找对应给组织机构数据
     * @param type 组织机构类型
     * @param orgId 组织机构ID
     * @return
     */
    @Override
    public OrgData getOrgDataById(String type, String orgId) {
        TreeModel<OrgData> tree = this.getOrgTree(type);
        if( tree != null){
            TreeNode<OrgData> treeNode = tree.getNode(orgId);
            if (treeNode != null){
                return treeNode.getData();
            }
        }
        return null;
    }
    
    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @return
     */
    @Override
    public List<OrgData> findChildrenData(String type, String orgId) {
        return this.findChildrenData(type,orgId,false);
    }

    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @param isAllChild
     * @return
     */
    @Override
    public List<OrgData> findChildrenData(String type, String orgId, boolean isAllChild) {
        TreeModel<OrgData> treeModel = this.getOrgTree(type);
        TreeNode<OrgData> node = treeModel.getNode(orgId);
        if (node != null){
            return node.getChildrenData(isAllChild);
        }
        return Collections.emptyList();
    }
    
    /**
     * 根据组织机构的类型和id查找对应给组织机构下的用户
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @return
     */
    @Override
    public OrgData getRootNode(String type) {
        TreeModel<OrgData> tree = this.getOrgTree(type);
        if (tree != null){
            TreeNode<OrgData> root = tree.getRoot();
            OrgData rootNode = root.getData();
            return rootNode;
        }
       return null;
    }

    /**
     * 根据登录名检索用户
     *
     * @param param
     * @return 单个对象
     */
    @Override
    public List<User> getUserLikeParam(String param) {
        List<User> userList = OrgDirector.USER.getVaules();
        List<User> users = new ArrayList<User>();
        if (userList != null && !userList.isEmpty()){
            for (User user : userList) {
                String userName = user.getUserName();
                boolean containName = StringUtil.hasValue(userName) && userName.contains(param);
                String loginName = user.getLoginName();
                boolean containLogin = StringUtil.hasValue(loginName) && loginName.contains(param);
                if (containName || containLogin) {
                    users.add(user);
                }
            }
        }
        return users;
    }
    /**
     * 根据传参值检索相似用户
     *
     * @param param
     * @return 单个对象
     */
    @Override
    public List<User> getUserLikeParam(String param, int searchMax) {
        List<User> users = this.getUserLikeParam(param);
        int size = users.size();
        if (searchMax <= 0||size == 0||size <= searchMax){
            return users;
        }else {
            return Lists.newArrayList(users.subList(0, searchMax));
        }
    }

    /**
     * 根据用户id检索用户
     *
     * @param id
     * @return 单个对象
     */
    @Override
    public User getUserById(String id) {
        List<User> users = OrgDirector.USER.getVaules();
        int index = users.indexOf(new User(id));
        User user = null;
        if (index >-1) {
            user = users.get(index);
        }
        return user;
    }

    /**
     * 根据用户id检索用户
     *
     * @return 单个对象
     */
    @Override
    public List<User> findAllUser() {
        return OrgDirector.USER.getVaules();
    }
    /**
     * 根据登录名检索用户
     *
     * @param loginName
     * @return 单个对象
     */
    @Override
    public User getUserByLoginName(String loginName) {
        List<User> users = OrgDirector.USER.getVaules();
        for (User user : users) {
            if (Objects.equal(user.getLoginName(), loginName)){
                return user;
            }
        }
        return null;
    }


    /**
     * 根据用户id查找用户所在的组织机构
     * @param userId
     * @return
     */
    @Override
    public List<OrgData> getOrgForUserByUserId(String userId) {
        List<OrgData> results = new ArrayList<OrgData>();
        List<OrgData> orgDatas = this.findOrgData();
        for (OrgData orgData : orgDatas) {
            List<User> users = orgData.getUsers();
            if (users.contains(new User(userId))){
                results.add(orgData);
            }
        }
        return results;
    }

    /**
     * 根据用户id和组织机构类型查找用户所在的组织机构
     * @param type
     * @param userId
     * @return
     */
    @Override
    public List<OrgData> getOrgForUserByUserId(String type, String userId) {
        List<OrgData> results = new ArrayList<OrgData>();
        TreeModel<OrgData> tree = this.getOrgTree(type);
        if (tree != null){
            List<OrgData> orgDatas = tree.getNodeDatas();
            for (OrgData orgData : orgDatas) {
                List<User> users = orgData.getUsers();
                if (users.contains(new User(userId))){
                    results.add(orgData);
                }
            }
        }
        return results;
    }

    /**
     * 根据组织机构的类型和id查找所有的上级
     * @param type type D:部门;G:群组;P:岗位;R:角色
     * @param orgId 组织机构id
     * @return
     */
    @Override
    public List<OrgData> findParentToRoot(String type, String orgId) {
        TreeModel<OrgData> tree = this.getOrgTree(type);
        List<OrgData> parentData = null;
        if (tree != null){
            TreeNode<OrgData> treeNode = tree.getNode(orgId);
            if (treeNode != null){
                parentData = treeNode.getDataPath();
            }
        }
        return parentData;
    }

    /**
     * 根据类型获得内存树
     *
     * @param type D:部门;G:群组;P:岗位;R:角色
     * @return
     */
    @Override
    public TreeModel<OrgData> getOrgTree(String type) {
        TreeModel<OrgData> treeModel = null;
        if (Objects.equal(DEPT, type)) {
            treeModel = OrgDirector.DEPT.tree();
        } else if (Objects.equal(GROUP, type)) {
            treeModel = OrgDirector.GROUP.tree();
        } else if (Objects.equal(POSITION, type)) {
            treeModel = OrgDirector.POSITION.tree();
        } else if (Objects.equal(ROLE, type)) {
            treeModel = OrgDirector.ROLE.tree();
        }
        return treeModel;
    }

    /**
     * 全部组织机构数据
     *
     * @return
     */
    @Override
    public List<OrgData> findOrgData(){

        List<OrgData> datas = Lists.newArrayList();
        TreeModel<OrgData> deptTree = OrgDirector.DEPT.tree();
        if (deptTree != null){
            List<OrgData> depts = deptTree.getNodeDatas();
            if (depts != null && !depts.isEmpty()) {
                datas.addAll(depts);
            }
        }
        TreeModel<OrgData> groupTree = OrgDirector.GROUP.tree();
        if (groupTree != null) {
            List<OrgData> groups = groupTree.getNodeDatas();
            if (groups != null && !groups.isEmpty()) {
                datas.addAll(groups);
            }

        }
        TreeModel<OrgData> positionTree = OrgDirector.POSITION.tree();
        if (positionTree != null) {
            List<OrgData> positions = positionTree.getNodeDatas();
            if (positions != null && !positions.isEmpty()) {
                datas.addAll(positions);
            }
        }
        TreeModel<OrgData> roleTree = OrgDirector.ROLE.tree();
        if (roleTree != null) {
            List<OrgData> roles = roleTree.getNodeDatas();
            if (roles != null && !roles.isEmpty()) {
                datas.addAll(roles);
            }

        }
        return datas;
    }
    /**
     * 根据类型获得对应的组织机构树
     *
     * @param type D:部门;G:群组;P:岗位;R:角色
     * @return
     */

    @Override
    public List<OrgData> findOrgData(String type){
        List<OrgData> datas = null;
        TreeModel<OrgData> tree = this.getOrgTree(type);
        if (tree != null){
            datas = tree.getNodeDatas();
        }
        return datas;
    }

}
