package com.horizon.third.init;

import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.horizon.common.collect.Maps;
import com.horizon.core.common.service.SpringContextHolder;
import com.horizon.core.init.HInit;
import com.horizon.core.init.HorizonInitAble;
import com.horizon.exception.extend.InitException;
import com.horizon.third.RelationUsers;
import com.horizon.third.adapter.WorkflowBaseAdapther;
import com.horizon.third.build.DefBuilder;
import com.horizon.third.build.OrgBuilder;
import com.horizon.third.build.OrgDirector;
import com.horizon.utils.ReflectHelper;
import com.horizon.utils.StringUtil;

/**
 * 用来进行实例初始化的工厂类<br/>
 * adapter初始化的类，注册到启动时<br/>
 * 
 * @author lichao 2016年9月21日
 */
@HInit
public class WorkflowAdapterInit implements HorizonInitAble {

    private static final Logger LOGGER = LoggerFactory.getLogger(WorkflowAdapterInit.class);

    // 缓存块

    /**
     * 接口适配器
     */
    private static Map<String, WorkflowBaseAdapther> workflowAdapthers;

    /**
     * 组织机构构建器
     */
    private static OrgBuilder organizationBuilder;

    /**
     * 相对关系
     */
    private static Map<String, RelationUsers> relations;

    /**
     * start 方法,容器启动过程中调用
     */
    @Override
    public void start() {

        workflowAdaptherInit();
        organizationInit();
        relationInit();
    }

    /**
     * 初始化接口适配器
     * 
     * @throws InitException
     */
    private void workflowAdaptherInit() throws InitException {

        LOGGER.info("  初始化流程接口适配器模块");
        Map<String, String> defaultMap = Maps.newLinkedHashMap();
        defaultMap.put("access", "com.horizon.third.adapter.PersistenceAdapter");
        defaultMap.put("calendar", "com.horizon.third.adapter.WorkCalendarAdapter");
        defaultMap.put("identity", "com.horizon.third.adapter.IdentityAdapter");
        defaultMap.put("tenant", "com.horizon.third.adapter.TenantAdapter");
        defaultMap.put("authority", "com.horizon.third.adapter.FlowManageAdapter");
        defaultMap.put("form", "com.horizon.third.adapter.FormAdapter");
        defaultMap.put("orgs", "com.horizon.third.adapter.OrganizationAdapter");
        defaultMap.put("session", "com.horizon.third.adapter.SessionAdapter");
        defaultMap.put("select", "com.horizon.third.adapter.SelectUserAdapter");

        // 1.2 初始化接口适配器实例化容器
        workflowAdapthers = Maps.newHashMap();
        initAdapters(defaultMap);
    }

    /**
     * 初始化组织机构数据
     * 
     * @throws InitException
     */
    private void organizationInit() throws InitException {

        LOGGER.info("  初始化流程组织机构模块");
        WorkflowAdapterFactory adapterFactory = getInstanceInitProperties();
        organizationBuilder = new DefBuilder();
        if (adapterFactory != null) {
            String orgClass = adapterFactory.getOrganizationBuilder();
            if (StringUtil.hasValue(orgClass)) {
                organizationBuilder = ReflectHelper.initClass(orgClass);
            }
        }

        OrgDirector.INSTANCE.build(organizationBuilder);
    }

    /**
     * 相对关系初始化
     * 
     * @throws InitException
     */
    private void relationInit() throws InitException {

        LOGGER.info("  初始化办理人相对关系模块");
        Map<String, String> defaultMap = Maps.newLinkedHashMap();
        defaultMap.put("1", "com.horizon.third.relation.SameDeptUser");
        defaultMap.put("2", "com.horizon.third.relation.Leaderhip");
        defaultMap.put("3", "com.horizon.third.relation.HigherLeaders");
        defaultMap.put("4", "com.horizon.third.relation.Subordinate");
        defaultMap.put("5", "com.horizon.third.relation.Subordinates");
        defaultMap.put("6", "com.horizon.third.relation.HigherLeader");
        defaultMap.put("9", "com.horizon.third.relation.HigherDeptAndSub");
        defaultMap.put("15", "com.horizon.third.relation.DeptLiaison");
        defaultMap.put("16", "com.horizon.third.relation.HigherDept");
        defaultMap.put("17", "com.horizon.third.relation.HigherPosition");
        defaultMap.put("18", "com.horizon.third.relation.HigherPositionUser");
        defaultMap.put("19", "com.horizon.third.relation.LowerPosition");
        defaultMap.put("20", "com.horizon.third.relation.LowerPositionUser");

        relations = Maps.newHashMap();
        initRelations(defaultMap);
    }

    /**
     * 销毁方法
     */
    @Override
    public void destory() {

        if (workflowAdapthers != null) {
            workflowAdapthers.clear();
        }
        if (organizationBuilder != null) {
            organizationBuilder.destory();
        }
        organizationBuilder = null;
        workflowAdapthers = null;
        relations = null;

    }

    @Override
    public int getSort() {

        return 12;
    }

    /**
     * 实例化适配器
     * 
     * @param defaultMap
     * @throws InitException
     */
    private void initAdapters(Map<String, String> defaultMap) {

        WorkflowAdapterFactory adapterFactory = getInstanceInitProperties();
        if (adapterFactory != null) {
            Map<String, String> adapthersMap = adapterFactory.getWorkflowAdaptherMap();
            if (adapthersMap != null && !adapthersMap.isEmpty()) {
                for (Entry<String, String> entry : adapthersMap.entrySet()) {
                    defaultMap.put(entry.getKey(), entry.getValue());
                }
            }
        }
        String key = null;
        String val = null;
        for (Entry<String, String> entry : defaultMap.entrySet()) {
            key = entry.getKey();
            val = entry.getValue();
            workflowAdapthers.put(key, (WorkflowBaseAdapther) ReflectHelper.initClass(val));
        }
    }

    /**
     * 实例化相对关系
     * 
     * @param defaultMap
     */
    private void initRelations(Map<String, String> defaultMap) {

        String key = null;
        String val = null;
        WorkflowAdapterFactory adapterFactory = getInstanceInitProperties();
        if (adapterFactory != null) {
            Map<String, String> relationMap = adapterFactory.getRelationMap();
            if (relationMap != null && !relationMap.isEmpty()) {
                for (Entry<String, String> entry : relationMap.entrySet()) {
                    defaultMap.put(entry.getKey(), entry.getValue());
                }
            }
        }

        for (Entry<String, String> entry : defaultMap.entrySet()) {
            key = entry.getKey();
            val = entry.getValue();
            relations.put(key, (RelationUsers) ReflectHelper.initClass(val));
        }
    }

    /**
     * 获取配置类
     * 
     */
    private WorkflowAdapterFactory getInstanceInitProperties() {

        return SpringContextHolder.getBean(WorkflowAdapterFactory.class);
    }

    /**
     * 获取接口适配器
     * 
     * @return
     */
    public static WorkflowBaseAdapther getWorkflowAdapthers(String key) {

        return workflowAdapthers != null ? workflowAdapthers.get(key) : null;
    }

    /**
     * 组织机构构建器
     * 
     * @return
     */
    public static OrgBuilder getOrganizationBuilder() {

        return organizationBuilder;
    }

    /**
     * 获取相对关系
     * 
     * @return
     */
    public static RelationUsers getRelations(String key) {

        return relations != null ? relations.get(key) : null;
    }

}
