package com.horizon.third.build;

import java.util.List;
import java.util.Map;

import com.horizon.common.base.Objects;
import com.horizon.core.tree.TreeModel;
import com.horizon.utils.HorizonPorps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.horizon.cache.api.HTenantCacheClient;
import com.horizon.core.context.Context;
import com.horizon.core.context.holder.ContextHolder;
import com.horizon.datasource.util.DataSourceContextHolder;
import com.horizon.third.TenantInfo;
import com.horizon.third.ThirdAdapterFactory;
import com.horizon.third.entity.OrgData;
import com.horizon.third.entity.User;
import com.horizon.utils.HorizonPool;

/**
 * 组织机构构建器
 * 
 * @author mawr
 *
 */
public abstract class OrgBuilder {

    private static final Logger LOGGER = LoggerFactory.getLogger(OrgBuilder.class);

    private static final String TENANTCODE = "tenantcode";

    /**
     * 部门
     * <p>
     * 由子类实现
     */
    public abstract List<OrgData> depts();

    /**
     * 群组
     * <p>
     * 由子类实现
     */
    public abstract List<OrgData> groups();

    /**
     * 岗位列表
     * <p>
     * 由子类实现
     * 
     * @return
     */
    public abstract List<OrgData> positions();

    /**
     * 岗位列表
     * <p>
     * 由子类实现
     * 
     * @return
     */
    public abstract List<OrgData> roles();

    /**
     * 用户
     * <p>
     * 由子类实现
     * 
     */
    public abstract List<User> users();

    /**
     * 销毁
     */
    public final void destory() {

        // 首先获得所有租户
        TenantInfo info = ThirdAdapterFactory.getTenantInfo();
        List<Map<String, String>> tenants = info.getTenants();
        for (Map<String, String> map : tenants) {
            String tenantCode = map.get(TENANTCODE);
            HTenantCacheClient.clear(tenantCode, HorizonPool.HORIZON_CACHE_VALUE);
        }

    }

    public final void init() {

        // 首先获得所有租户
        TenantInfo info = ThirdAdapterFactory.getTenantInfo();
        List<Map<String, String>> tenants = info.getTenants();
        String dataSource = DataSourceContextHolder.getDataSourceName();

        initDeptTree(tenants);
        initGroupTree(tenants);
        initPositionTree(tenants);
        initRoleTree(tenants);
        initUsers(tenants);

        DataSourceContextHolder.setDataSourceType(dataSource);
    }

    /**
     * 初始用户
     * 
     * @param tenants
     */
    public final void initUsers(List<Map<String, String>> tenants) {
        String loadAll = HorizonPorps.getConfig("orgData.loadAll");
        if (!Objects.equal("false", loadAll)) {
            long begin = System.currentTimeMillis();
            for (Map<String, String> map : tenants) {
                Context context = ThirdAdapterFactory.switchTenant(map.get(TENANTCODE));
                HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                        HorizonPool.HORIZON_CACHE_USER, users(), 0L);
                ContextHolder.setContext(context);
            }
            long end = System.currentTimeMillis();
            LOGGER.info(" Init users use timer {} s", (end - begin) / 1000);
        }

    }

    /**
     * 初始化部门
     * 
     * @param tenants
     * 
     * @return
     */

    public final void initDeptTree(List<Map<String, String>> tenants) {
        String loadAll = HorizonPorps.getConfig("orgData.loadAll");
        boolean isLoadAll = !Objects.equal("false", loadAll);
        long begin = System.currentTimeMillis();
        for (Map<String, String> map : tenants) {
            Context context = ThirdAdapterFactory.switchTenant(map.get(TENANTCODE));
            try {
                List<OrgData> depts = depts();
                if (isLoadAll) {
                    TreeModel<OrgData> tree = this.buildTree(depts);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_DEPT, tree, 0L);
                }else {
                    initOrgData(depts);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_DEPT, depts, 0L);
                }

            } catch (Exception e) {
                LOGGER.warn("租户:" + map.get(TENANTCODE) + "部门初始化失败:" + e.getMessage());
            }
            ContextHolder.setContext(context);
        }
        long end = System.currentTimeMillis();
        LOGGER.info(" Init depts use timer {} s", (end - begin) / 1000);
    }

    /**
     * 初始化群组
     * 
     * @param tenants
     * 
     * @return
     */
    public final void initGroupTree(List<Map<String, String>> tenants) {
        String loadAll = HorizonPorps.getConfig("orgData.loadAll");
        boolean isLoadAll = !Objects.equal("false", loadAll);
        long begin = System.currentTimeMillis();
        for (Map<String, String> map : tenants) {
            Context context = ThirdAdapterFactory.switchTenant(map.get(TENANTCODE));
            try {
                List<OrgData> groups = groups();
                if (isLoadAll){
                    TreeModel<OrgData> tree = buildTree(groups);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_GROUP, tree, 0L);
                }else {
                    initOrgData(groups);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_GROUP, groups, 0L);
                }

            } catch (Exception e) {
                LOGGER.warn("租户:" + map.get(TENANTCODE) + "群组初始化失败:" + e.getMessage());
            }
            ContextHolder.setContext(context);
        }
        long end = System.currentTimeMillis();
        LOGGER.info(" Init groups use timer {} s", (end - begin) / 1000);
    }

    /**
     * 初始化岗位
     * 
     * @return
     */
    public final void initPositionTree(List<Map<String, String>> tenants) {
        String loadAll = HorizonPorps.getConfig("orgData.loadAll");
        boolean isLoadAll = !Objects.equal("false", loadAll);
        long begin = System.currentTimeMillis();
        for (Map<String, String> map : tenants) {
            Context context = ThirdAdapterFactory.switchTenant(map.get(TENANTCODE));
            try {
                List<OrgData> positions = positions();
                if (isLoadAll) {
                    TreeModel<OrgData> tree = buildTree(positions);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_POSITION, tree, 0L);
                }else {
                    initOrgData(positions);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_POSITION, positions, 0L);
                }

            } catch (Exception e) {
                LOGGER.warn("租户:" + map.get(TENANTCODE) + "岗位初始化失败:" + e.getMessage());
            }
            ContextHolder.setContext(context);
        }
        long end = System.currentTimeMillis();
        LOGGER.info(" Init positions use timer {} s", (end - begin) / 1000);
    }

    /**
     * 初始化角色
     * 
     * @return
     */
    public final void initRoleTree(List<Map<String, String>> tenants) {
        String loadAll = HorizonPorps.getConfig("orgData.loadAll");
        boolean isLoadAll = !Objects.equal("false", loadAll);
        long begin = System.currentTimeMillis();
        for (Map<String, String> map : tenants) {
            Context context = ThirdAdapterFactory.switchTenant(map.get(TENANTCODE));
            try {
                List<OrgData> roles = roles();
                if (isLoadAll) {
                    TreeModel<OrgData> tree = buildTree(roles);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_ROLE, tree, 0L);
                }else {
                    initOrgData(roles);
                    HTenantCacheClient.put(map.get(TENANTCODE), HorizonPool.HORIZON_CACHE_VALUE,
                            HorizonPool.HORIZON_CACHE_ROLE, roles, 0L);
                }

            } catch (Exception e) {
                LOGGER.warn("租户:" + map.get(TENANTCODE) + "角色初始化失败:" + e.getMessage());
            }
            ContextHolder.setContext(context);
        }
        long end = System.currentTimeMillis();

        LOGGER.info(" Init roles use timer {} s", (end - begin) / 1000);
    }
    /**
     * 构建树
     *
     * @param datas
     * @return
     */
    protected TreeModel<OrgData> buildTree(List<OrgData> datas) {

        TreeModel<OrgData> tree = null;
        if (datas != null) {
            tree = new TreeModel<OrgData>().creat(datas);
        }
        return tree;
    }
    /**
     * 不需要加载组织机构用户时,不组装用户
     * @param orgDatas
     */
    private void initOrgData(List<OrgData> orgDatas){
        if (orgDatas != null && !orgDatas.isEmpty()){
            for (OrgData orgData : orgDatas) {
                List<User> users = orgData.getUsers();
                if (users != null ){
                    orgData.setUsers(null);
                }
            }
        }
    }


}
