package com.horizon.third.adapter;

import com.horizon.common.base.Joiner;
import com.horizon.common.base.Objects;
import com.horizon.common.base.Preconditions;
import com.horizon.common.base.Splitter;
import com.horizon.common.collect.Lists;
import com.horizon.common.collect.Maps;
import com.horizon.common.collect.Sets;
import com.horizon.core.tree.TreeModel;
import com.horizon.third.Organization;
import com.horizon.third.RelationUsers;
import com.horizon.third.entity.OrgData;
import com.horizon.third.entity.User;
import com.horizon.third.relation.RelationFactory;
import com.horizon.utils.StringUtil;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 组织机构接口适配器
 *
 * @author mwr
 */
public class OrganizationAdapter extends WorkflowBaseAdapther implements Organization {

    private static final String NOUSERID = "请求的参数 userId 为 null ";

    /**
     * 去掉orgs中不存在的组织机构id
     * @param orgs  组织机构id字符串
     *
     * @return
     */
    @Override
    public String removeNotExitsOrg(String orgs, String identifier) {

        Preconditions.checkNotNull(orgs, "请求的参数 orgs 为 null ");
        List<String> ids = Lists.newArrayList();
        List<String> orgIds = Splitter.on(SPLITCHAR).omitEmptyStrings().splitToList(orgs);
        for (String orgId : orgIds) {
            if (!orgIsExits(orgId)) {
                ids.add(orgId);
            }
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    /**
     * 根据给定的部门id获取部门下的用户
     *
     * @param deptId
     *            多个部门id以";"隔开
     * @param isHaveSubDept
     *            是否含有子部门;
     *            <p>
     *            true:含有子部门;
     *            <p>
     *            false：不含子部门
     * @param identifier
     *            数据源标识符
     * @return
     */
    @Override
    public String getAllUserFromDept(String deptId, boolean isHaveSubDept, String identifier) {
        Preconditions.checkNotNull(deptId, "请求的参数 deptId 为 null ");
        return getAllUserFromDept(deptId, isHaveSubDept, false, identifier);
    }

    /**
     * 根据给定的部门id获取部门下的用户
     *
     * @param deptId
     *            多个部门id以";"隔开
     * @param isHaveSubDept
     *            是否含有子部门;
     *            <p>
     *            true:含有子部门;
     *            <p>
     *            false：不含子部门
     * @param allNodes
     *            含有子部门;
     *            <p>
     *            true:含有直接子部门;
     *            <p>
     *            false：全部子部门
     * @param identifier
     *            数据源标识符
     * @return
     */
    public String getAllUserFromDept(String deptId, boolean isHaveSubDept, boolean allNodes, String identifier) {

        Preconditions.checkNotNull(deptId, "请求的参数 deptid 为 null ");
        Set<String> ids = Sets.newTreeSet();
        List<String> deptIds = Splitter.on(SPLITCHAR).omitEmptyStrings().splitToList(deptId);
        for (String id : deptIds) {
            if (isHaveSubDept) {
                List<OrgData> subDeptData = super.findChildrenData(DEPT, id, allNodes);
                for (OrgData subDept : subDeptData) {
                    List<User> users = super.getUserByOrgId(DEPT, subDept.getId());
                    if (users != null && !users.isEmpty()){
                        for (User user : users) {
                            ids.add(user.getId());
                        }
                    }
                }
            }
            List<User> users = super.getUserByOrgId(DEPT, id);
            if (users != null && !users.isEmpty()){
                for (User user : users) {
                    ids.add(user.getId());
                }
            }
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    /**
     * 根据给定的群组id获取群组下的用户
     *
     * @param groupId
     *            ：多个部门id以";"隔开
     * @param identifier
     * @return 用户id组成的字符串,多个以";"隔开
     */
    @Override
    public String getAllUserFromGroup(String groupId, String identifier) {

        Preconditions.checkNotNull(groupId, "请求的参数 groupid 为 null ");
        List<String> groupIds = Splitter.on(SPLITCHAR).omitEmptyStrings().splitToList(groupId);
        Set<String> ids = Sets.newTreeSet();
        for (String id : groupIds) {
            List<User> users = super.getUserByOrgId(GROUP, id);
            if (users != null && !users.isEmpty()){
                for (User user : users) {
                    ids.add(user.getId());
                }
            }
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    /**
     * 根据给定的岗位id获取岗位下的用户
     *
     * @param positionId
     *            ：多个岗位id以";"隔开
     * @param identifier
     * @return 用户id组成的字符串,多个以";"隔开
     */
    @Override
    public String getAllUserFromPosition(String positionId, String identifier) {

        Preconditions.checkNotNull(positionId, "请求的参数 postionId 为 null ");
        List<String> positionIds = Splitter.on(SPLITCHAR).omitEmptyStrings().splitToList(positionId);
        Set<String> ids = Sets.newTreeSet();
        for (String id : positionIds) {
            List<User> users = super.getUserByOrgId(POSITION, id);
            if (users != null && !users.isEmpty()){
                for (User user : users) {
                    ids.add(user.getId());
                }
            }
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    /**
     * 根据组织机构中的人员之间的关系获取人员(比如根据userid来获取直接领导,下属)
     *
     * @param para
     *            <p>
     *            key = userID value = String 流程当前办理用户id
     *            <p>
     *            key = creator value = String 流程创建用户id
     *            <p>
     *            key = done value = String[] 目标节点历史办理人id
     *            <p>
     *            key = base value = int[] 基准 0=无,1=指定节点办理人,2=创建者,3=当前执行人curUser
     *            <p>
     *            key = tarRange value = String[] 相对岗位
     *            <p>
     *            key = gathertype value = int[] 交集并集默认2并集
     *            <p>
     *            key = relation value = String[] 关系编号
     * @param identifier
     *            数据源标识符
     * @return
     */
    @Override
    public String getRelationUser(Map<String, Object> para, String identifier) {

        String authorId = null;
        String[] relations = (String[]) para.get("relation");
        int[] base = (int[]) para.get("base");
        List<String> ids = Lists.newArrayList();
        for (int i = 0, iLen = base.length; i < iLen; i++) {
            switch (base[i]) {
                case 1:
                    // 已办人
                    authorId = ((String[]) para.get("done"))[i];
                    break;
                case 2:
                    // 流程启动者
                    authorId = (String) para.get("creator");
                    break;
                case 3:
                    // 当前执行人
                    authorId = (String) para.get("userid");
                    break;
                default:
                    break;
            }
            List<String> idList = Splitter.on(";").omitEmptyStrings().splitToList(authorId);
            String[] idArray = idList.toArray(new String[idList.size()]);
            RelationUsers relationUsers = RelationFactory.INSTANCE.create(relations[i]);
            List<String> users = relationUsers.getRelations(idArray);
            ids.addAll(users);
        }
        return Joiner.on(SPLITCHAR).skipNulls().join(ids);
    }

    /**
     * 获取给定用户的所有相关id
     *
     * @param userId
     *            用户ID
     * @param identifier
     *            数据源标识符
     * @return
     */
    @Override
    public String getAllUserids(String userId, String identifier) {
        User user = super.getUserById(userId);
        String parentId = null;
        String otherDeptId = null;
        if (user != null ) {
            otherDeptId = this.getOtherDeptIds(user);
            parentId = super.getParentId(user.getFirstDept());
        }
        String orgId = this.getAllOrgDataIds(userId);
        return Joiner.on(",").skipNulls().join(userId, orgId, parentId, otherDeptId);
    }
    /**
     * 根据用户ID获得用户所属的组织机构ID(部门,群组,岗位,角色)
     *
     * @param userId 用户ID
     * @return
     */
    protected String getAllOrgDataIds(String userId) {
        Set<String> ids = Sets.newTreeSet();
        List<OrgData> orgDatas= this.getOrgForUserByUserId(userId);
        for (OrgData orgData : orgDatas) {
            ids.add(orgData.getId());
        }
        return Joiner.on(",").join(ids.iterator());

    }



    /**
     * 获取给定部门deptId的下级子部门id
     *
     * @param userId
     *            部门Id
     * @param identifier
     *            租户标识
     * @return
     */
    @Override
    public Map<String, String> getUserById(String userId, String identifier) {

        Preconditions.checkNotNull(userId, NOUSERID);
        List<Map<String, String>> users = this.getUserByIDs(new String[]{userId}, identifier);
        return users.get(0);
    }


    /**
     * 获取给定部门deptId的下级子部门id
     *
     * @param deptId
     *            部门Id
     * @param identifier
     *            租户标识
     * @return
     */
    @Override
    public String getSubDeptIdById(String deptId, String identifier) {

        Preconditions.checkNotNull(deptId, "请求的参数 deptid 为 null ");
        List<String> deptIds = Splitter.on(SPLITCHAR).omitEmptyStrings().splitToList(deptId);
        Set<String> ids = Sets.newTreeSet();
        for (String id : deptIds) {
            List<OrgData> orgDatas = super.findChildrenData(DEPT, id);
            if (orgDatas != null && !orgDatas.isEmpty()) {
                for (OrgData orgData : orgDatas) {
                    ids.add(orgData.getId());
                }
            }
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }


    /**
     * 获取指定userid所在的直属部门id
     *
     * @param userId
     *            用户ID
     * @param identifier
     *            租户标识
     * @return
     */
    @Override
    public String getDeptIdByUserid(String userId, String identifier) {
        Preconditions.checkNotNull(userId, NOUSERID);
        User user = super.getUserById(userId);
        return user.getFirstDept();
    }

    /**
     * 获取指定id的中文名称,id可能是任何组织类型
     *
     * @param id
     * @param identifier
     *            数据源标识符
     * @return
     */
    @Override
    public String getNameByIds(String id, String identifier) {

        Preconditions.checkNotNull(id, "请求的参数 id 为 null ");
        List<String> names = Lists.newArrayList();
        List<String> orgIds = Splitter.on(SPLITCHAR).omitEmptyStrings().splitToList(id);
        for (String orgId : orgIds) {
            List<String> fixIds = Splitter.on(SUBJECTION_FIX).trimResults().omitEmptyStrings().splitToList(orgId);
            if (fixIds.size() == 1 ){
                String tempId = fixIds.get(0);
                String userName = this.getUserName(tempId);
                if (StringUtil.hasValue(userName)) {
                    names.add(userName);
                } else{
                    String orgName = this.getOrgName(tempId);
                    if (StringUtil.hasValue(orgName)){
                        names.add(orgName);
                    }
                }
            }else {
                String userId = fixIds.get(0);
                String deptId = fixIds.get(1);
                String userName =  this.getUserName(userId);
                String orgName =  this.getOrgName(deptId);
                String name = StringUtil.hasValue(userName)? userName + "(" + orgName + ")":"(" + orgName + ")";
                names.add(name);
            }

        }
        if (!names.isEmpty()){
            return Joiner.on(SPLITCHAR).join(names);
        }
       return null;
    }
    /**
     * 根据用户Id获得用户信息 设计器用到的方法
     *
     * @param userIds
     * @return List<Map> map中的key: id, name
     *
     */
    @Override
    public List<Map<String, String>> getUserByIDs(String[] userIds, String identifier) {
        Preconditions.checkNotNull(userIds, "请求的参数 userIds 为 null ");
        List<Map<String, String>> lstUsers = Lists.newArrayList();
        for (String userId : userIds) {
            User user = super.getUserById(userId);
            if (user != null) {
                Map<String, String> map = Maps.newHashMap();
                map.put("id", user.getId());
                map.put("name", user.getUserName());
                map.put("login_name", user.getLoginName());
                map.put("dept_id", user.getFirstDept());
                lstUsers.add(map);
            }
        }
        return lstUsers;
    }
    /**
     * 根据部门Id获得部门信息
     *
     * @param deptIds
     *            部门Id数组
     * @return List<Map> map中的key: id,dept_name
     */
    @Override
    public List<Map<String, String>> getDeptsByIDs(String[] deptIds, String identifier) {

        Preconditions.checkNotNull(deptIds, "请求的参数 deptIds 为 null ");
        List<Map<String, String>> lstDepts = Lists.newArrayList();
        List<String> deptIdList = Arrays.asList(deptIds);
        for (String deptId : deptIdList) {
            OrgData data = super.getOrgDataById(DEPT, deptId);
            if (data != null){
                String id = data.getId();
                if (deptIdList.contains(id)){
                    Map<String, String> map = Maps.newHashMap();
                    map.put("id", deptId);
                    map.put("dept_name", data.getName());
                    lstDepts.add(map);
                }
            }
        }
        return lstDepts;

    }
    /**
     * 根据群组Id获得群组信息 设计器用到的方法
     *
     * @param groupIds
     *            群组Id数组
     * @return List<Map> map中的key: id,group_name
     */
    @Override
    public List<Map<String, String>> getGroupByIDs(String[] groupIds, String identifier) {

        Preconditions.checkNotNull(groupIds, "请求的参数 groupIds 为 null ");
        List<String> groupIdList = Arrays.asList(groupIds);
        List<Map<String, String>> lstGroups = Lists.newArrayList();
        for (String groupId : groupIdList) {
            OrgData data = super.getOrgDataById(GROUP, groupId);
            if (data != null){
                String id = data.getId();
                if (groupIdList.contains(id)){
                    Map<String, String> map = Maps.newHashMap();
                    map.put("id", groupId);
                    map.put("group_name", data.getName());
                    lstGroups.add(map);
                }
            }
        }
        return lstGroups;
    }
    /**
     * 根据岗位Id获得岗位信息 设计器用到的方法
     *
     * @param positionIds
     * @return List<Map> map中的key: id,position_name
     */
    @Override
    public List<Map<String, String>> getPositionByIDs(String[] positionIds, String identifier) {
        Preconditions.checkNotNull(positionIds, "请求的参数 positionIds 为 null ");
        List<String> positionIdList = Arrays.asList(positionIds);
        List<Map<String, String>> lstPositions = Lists.newArrayList();
        for (String positionId : positionIdList) {
            OrgData data = super.getOrgDataById(POSITION, positionId);
            if (data != null){
                String id = data.getId();
                if (positionIdList.contains(id)){
                    Map<String, String> map = Maps.newHashMap();
                    map.put("id", positionId);
                    map.put("position_name", data.getName());
                    lstPositions.add(map);
                }
            }
        }
        return lstPositions;
    }
    /**
     * 获取指定userid所在的直属群组id
     *
     * @param userId
     *            用户ID
     * @param identifier
     *            租户标识
     * @return
     */
    @Override
    public String getGroupIdByUserid(String userId, String identifier) {
        Preconditions.checkNotNull(userId, NOUSERID);
        List<OrgData> orgDatas = super.getOrgForUserByUserId(GROUP, userId);
        Set<String> ids = Sets.newTreeSet();
        for (OrgData orgData : orgDatas) {
            ids.add(orgData.getId());
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    /**
     * 获取指定userId所在的直属岗位id
     *
     * @param userId
     *            用户ID
     * @param identifier
     *            数据源标识符
     * @return
     */
    @Override
    public String getPositionIdByUserid(String userId, String identifier) {

        Preconditions.checkNotNull(userId, NOUSERID);
        List<OrgData> orgDatas = super.getOrgForUserByUserId(POSITION, userId);
        Set<String> ids = Sets.newTreeSet();
        for (OrgData orgData : orgDatas) {
            ids.add(orgData.getId());
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    /**
     * 获取指定userId所在的直属角色id
     *
     * @param userId
     *            用户ID
     * @param identifier
     *            数据源标识符
     * @return
     */
    @Override
    public String getRoleIdByUserid(String userId, String identifier) {
        List<OrgData> orgDatas = super.getOrgForUserByUserId(ROLE, userId);
        Set<String> ids = Sets.newTreeSet();
        for (OrgData orgData : orgDatas) {
            ids.add(orgData.getId());
        }
        return Joiner.on(SPLITCHAR).join(ids.iterator());
    }

    @Override
    public Map<String, String> getUserByLoginName(String loginName, String identifier) {

        Preconditions.checkNotNull(loginName, "请求的参数 loginName 为 null ");
        User user = super.getUserByLoginName(loginName);
        Map<String, String> map = Maps.newHashMap();
        map.put("user_id", user.getId());
        map.put("user_name", user.getUserName());
        map.put("dept_id", user.getFirstDept());
        map.put("password", user.getPasswd());
        map.put("dept_name", getOrgName(user.getFirstDept()));
        return map;
    }
    /**
     * 获取兼职部门ID,如果不需要计算兼职部门将返回值设置为null
     *
     * @param user 用户对象
     * @return
     */
    private String getOtherDeptIds(User user) {

        String otherDeptId = null;
        List<String> deptIds = user.getDepartment();
        if (deptIds != null && deptIds.size() > 1) {
            List<String> subDeptIds = Lists.newArrayList(deptIds.subList(1, deptIds.size()));
            otherDeptId = Joiner.on(",").skipNulls().join(subDeptIds);
        }
        return otherDeptId;
    }

    /**
     * @param userId 用户ID
     * @return
     * @modifier chengll 2018年1月30日 根据用户ID获得用户所属的组织机构列表(部门,群组,岗位,角色)
     */
    public List<OrgData> getAllOrgsByUserId(String userId) {
        return super.getOrgForUserByUserId(userId);
    }



    /**
     * 根据ID获取用户名称
     *
     * @param userId 用户ID
     * @return
     */
    protected String getUserName(String userId) {
        List<String> names = Lists.newArrayList();
        List<String> userIds = Splitter.on(AGENT_SPLITCHAR).trimResults().omitEmptyStrings().splitToList(userId);
        if (!userId.isEmpty()){
            for (String id : userIds) {
                String name = this.getName(this.trimSuffix(id));
                if (StringUtil.hasValue(name)){
                    names.add(name);
                }
            }
        }
        if (!names.isEmpty()){
            return Joiner.on(AGENT_SPLITCHAR).join(names);
        }
        return null;
    }

    /**
     * 删除userId的前缀
     *
     * @param userId
     * @return
     */
    private String trimSuffix(String userId) {
        if (StringUtil.hasValue(userId)){
            userId= userId.replaceAll("\\]", "").replaceAll("\\[", "");
            userId = userId.startsWith(USER_FIX)? userId.substring(2):userId;
        }
        return userId;
    }
    /**
     * 根据部门ID获取部门名称
     *
     * @param id
     * @return
     */
    protected String getOrgName(String id) {
        Preconditions.checkNotNull(id, "请求参数 组织机构id 为null ");
        String orgName = null;
        if (!id.startsWith(USER_FIX)) {
            String orgId = null;
            String dataType = null;
            List<String> fixIds = Splitter.on("_").trimResults().omitEmptyStrings().splitToList(id);
            if (fixIds.size() == 1) {
                dataType = DEPT;
                orgId = id;
            }else if (fixIds.size() == 2) {
                dataType = fixIds.get(0);
                orgId = id.substring(2);
                dataType = Objects.equal(dataType, "S") ? DEPT : dataType;
            }
            OrgData data = super.getOrgDataById(dataType, orgId);
            if (data != null){
                orgName = data.getName();
            }
        }
        return orgName;
    }

    /**
     * 判断组织机构ID是否存在
     *
     * @param orgId
     * @return
     */
    protected boolean orgIsExits(String orgId) {

        List<String> idList = Splitter.on(SUBJECTION_FIX).omitEmptyStrings().splitToList(orgId);
        int size = idList.size();
        if (size == 1) {
            String id = idList.get(0);
            String type = getType(id);
            if (Objects.equal(type, "U")) {
                return isExistUserId(id);
            } else {
                return isExistOrgId(type,id);
            }
        } else {
            String userId = idList.get(0);
            String subjId = idList.get(1);
            String type = getType(subjId);
            if (!isExistUserId(userId)) {
                return false;
            }
            if (!isExistOrgId(type,subjId)) {
                return false;
            }
        }
        return false;
    }

    private String getType(String id) {
        if(StringUtil.hasValue(id)){
            int index = id.indexOf('_');
            if (index > -1){
                return id.substring(0, index);
            }

        }
        return null;
    }

    /**
     * 判断给定的组织机构ID是否存在
     * <p>
     * 不存在返回 true
     *
     * @param id 组织机构ID
     * @return
     */
    private boolean isExistOrgId(String type, String id) {
        String tempId = super.subPrefix(id);
        OrgData orgData = super.getOrgDataById(type, tempId);
        return orgData != null;
    }

    /**
     * 根据ID获得用户名称,如果用户不存在返回用户ID
     *
     * @param id 用户ID
     * @return
     */
    private String getName(String id) {
        User user = super.getUserById(id);
        if (user != null){
            return user.getUserName();
        }
       return null;
    }


    /**
     * 判断给定的用户ID是否存在
     * <p>
     * 不存在返回 true
     *
     * @param userId
     * @return
     */
    private boolean isExistUserId(String userId) {

        Preconditions.checkNotNull(userId, NOUSERID);
        String uid = super.subPrefix(userId);
        User user =super.getUserById(uid);
        return user != null;
    }

    /**
     * 判断是否开启权限过滤树
     *
     * @param tree 组织机构树
     * @param auth 权限开关
     * @return
     */
    @Override
    public TreeModel<OrgData> getTreeNodeByAuth(TreeModel<OrgData> tree, Boolean auth) {
        return tree;
    }

    /**
     * 判断是否开启权限过滤树
     *
     * @param orgDatas 组织机构数据
     * @param auth 权限开关
     * @return
     */
    @Override
    public List<OrgData> getTreeNodeByAuth(List<OrgData> orgDatas, Boolean auth) {
        return orgDatas;
    }

    /**
     * 根据deptid以及职务类型查询指定的用户id
     * @param deptId
     * @param type 职务类型
     * @param identifier
     * @return
     */
    @Override
    public String getDeptSpecialUser(String deptId, String type, String identifier) {
        List<User> orgUsers = super.findUserByOrgId(DEPT, deptId);
        User user = filterManageUser(type, orgUsers);
        if (user != null){
            return user.getId();
        }
        return null;
    }
    /**
     * 根据给定的用户列表，过滤具有管理职务的用户列表
     *
     * @param parm
     *            管理职务类型
     * @param users
     *            用户列表
     * @return
     */
    protected User filterManageUser(String parm, final List<User> users) {
        List<User> allUser = this.findAllUser();
        for (User user : users) {
            int index = allUser.indexOf(user);
            if ( index > -1) {
                User u = allUser.get(index);
                String operType = u.getOperType();
                if ( StringUtil.hasValue(operType) && operType.contains(parm)) {
                    return user;
                }
            }
        }
        return null;
    }
}
