package com.horizon.third;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.List;
import java.util.Map;

import com.horizon.common.base.Objects;
import com.horizon.common.collect.Lists;
import com.horizon.common.collect.Maps;
import com.horizon.core.HorizonCore;
import com.horizon.core.context.Context;
import com.horizon.core.context.ContextFactory;
import com.horizon.core.context.holder.ContextHolder;
import com.horizon.datasource.util.DataSourceContextHolder;
import com.horizon.db.DBConfigInfoHelper;
import com.horizon.third.adapter.WorkflowBaseAdapther;
import com.horizon.third.init.WorkflowAdapterInit;
import com.horizon.utils.HorizonPool;

/**
 * 适配器静态工程类
 *
 * @author mwr
 */
public final class ThirdAdapterFactory {

    private String tenantCode;

    private static Map<String, ThirdAdapterFactory> pools = Maps.newHashMap();

    private ThirdAdapterFactory(String tenantCode) {
        this.tenantCode = tenantCode;
    }

    public static ThirdAdapterFactory instanse(String tenantCode) {

        ThirdAdapterFactory factory = pools.get(tenantCode);
        if (factory == null) {
            factory = new ThirdAdapterFactory(tenantCode);
            pools.put(tenantCode, factory);
        }
        return factory;
    }

    /**
     * 获得持久化接口
     *
     * @param tenantCode
     * @return
     */
    public static Persistence getPersistenceInstance(String tenantCode) {

        return (Persistence) getAdapther("access");
    }

    /**
     * 获得持久化接口
     *
     * @return
     */
    public static Persistence getPersistenceInstance() {

        return getPersistenceInstance(null);
    }

    public static SessionInfo getSessionInfo() {

        return (SessionInfo) getAdapther("session");
    }

    public static TenantInfo getTenantInfo() {

        return (TenantInfo) getAdapther("tenant");
    }

    public static Form getFormInstance() {

        return (Form) getAdapther("form");
    }

    /**
     * 获得全部租户数据源标识
     *
     * @return
     */
    public static List<String> getAllIdentifierList() {
        List<String> list = Lists.newArrayList();
        TenantInfo info = getTenantInfo();
        List<Map<String, String>> maps = info.getTenants();
        if (maps != null && !maps.isEmpty()) {
            for (Map<String, String> map : maps) {
                list.add(map.get("dbidentifierflag"));
            }
        }
        String identifier = getDefaultDatabaseIdentifier();
        if (!list.contains(identifier)) {
            list.add(identifier);
        }
        return list;
    }

    /**
     * 根据租户表示获得组织机构接口
     *
     * @return
     */
    public Organization getOrganizationInstance() {

        return getAdaptherSwitchTenant("orgs");
    }

    public static Organization getOrganizationInstance(String tenantCode) {

        return ThirdAdapterFactory.instanse(tenantCode).getOrganizationInstance();
    }

    /**
     * 获得工作日历接口
     *
     * @return
     */
    public WorkCalendar getCalendarInstance() {

        return getAdaptherSwitchTenant("calendar");
    }

    public static WorkCalendar getCalendarInstance(String tenantCode) {

        return ThirdAdapterFactory.instanse(tenantCode).getCalendarInstance();
    }

    /**
     * 获得流程权限接口
     *
     * @return
     */
    public Authority getAuthorityInstance() {

        return getAdaptherSwitchTenant("authority");
    }

    public static Authority getAuthorityInstance(String tenantCode) {

        return ThirdAdapterFactory.instanse(tenantCode).getAuthorityInstance();
    }

    /**
     * 获得身份认证接口
     *
     * @return
     */
    public Identity getIdentityInstance() {

        return getAdaptherSwitchTenant("identity");
    }

    public static Identity getIdentityInstance(String tenantCode) {

        return ThirdAdapterFactory.instanse(tenantCode).getIdentityInstance();
    }

    public SelectUser getSelectUser() {

        return getAdaptherSwitchTenant("select");
    }

    public static SelectUser getSelectUser(String tenantCode) {

        return ThirdAdapterFactory.instanse(tenantCode).getSelectUser();
    }

    public String getTenantCode() {

        return tenantCode;
    }

    /**
     * 根据租户表示获得数据源标识
     *
     * @param tenantCode
     * @return
     */
    public static String getIdentifierFromTenantid(String tenantCode) {

        TenantInfo info = getTenantInfo();
        Map<String, String> tenant = info.getTenantName(tenantCode);
        if (tenant == null) {
            return null;
        }
        return tenant.get("dbidentifierflag");

    }

    /**
     * 根据数据源标识获得租户标识
     *
     * @param identifier
     * @return
     */
    public static String getTenantidByIdentifier(String identifier) {

        // TODO 这里的代码不对
        return getIdentifierFromTenantid(identifier);
    }

    /**
     * 获取adapter
     *
     */
    private static WorkflowBaseAdapther getAdapther(String key) {

        return WorkflowAdapterInit.getWorkflowAdapthers(key);
    }

    /**
     * 获取需要自动切换租户的adapter
     *
     */
    @SuppressWarnings("unchecked")
    private <T> T getAdaptherSwitchTenant(String key) {

        WorkflowBaseAdapther adapter = WorkflowAdapterInit.getWorkflowAdapthers(key);
        // 使用java的动态代理来进行切换租户和数据源
        return (T) Proxy.newProxyInstance(adapter.getClass().getClassLoader(), adapter.getClass().getInterfaces(),
                new SwitchTenantProxyHandler(adapter, tenantCode));
    }

    /**
     * 切换租户
     *
     * @return
     */
    public static Context switchTenant(String tenantCode) {

        // 先获取当前的context
        Context context = ContextFactory.INSTANCE.getContext();
        // 如果当前context不需要切换
        if (context != null && Objects.equal(context.getTenantCode(), tenantCode)) {
            return context;
        }
        Map<String, Object> attribute = Maps.newHashMap();
        // 如果是默认的数据源
        if (Objects.equal(tenantCode, DBConfigInfoHelper.getDefaultDatabaseIdentifier())) {
            attribute.put(HorizonPool.SESSION_TENANT_CODE, DBConfigInfoHelper.getDefaultDatabaseIdentifier());
            attribute.put(HorizonPool.SESSION_TENANT_IDENTIFIER, DBConfigInfoHelper.getDefaultDatabaseIdentifier());
        } else {
            attribute.put(HorizonPool.SESSION_TENANT_CODE, tenantCode);
            attribute.put(HorizonPool.SESSION_TENANT_IDENTIFIER, getIdentifierFromTenantid(tenantCode));
        }
        Context newContext = new Context();
        newContext.regiestContextAttribute(attribute);
        ContextHolder.setContext(newContext);
        DataSourceContextHolder.setDataSourceType(HorizonCore.TENANT_IDENTIFIER.value());
        return context;
    }

    /**
     * 切换数据源的动态代理类
     */
    private static class SwitchTenantProxyHandler implements InvocationHandler {

        private Object proxied;

        private String tenantCode;

        public SwitchTenantProxyHandler(Object proxied, String tenantCode) {
            this.proxied = proxied;
            this.tenantCode = tenantCode;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {

            // 切换租户，返回之前的状态
            String dataSource = DataSourceContextHolder.getDataSourceName();
            Context context = switchTenant(tenantCode);
            Object obj = null;

            try {
                obj = method.invoke(proxied, args);
            } catch (InvocationTargetException ex) {
                throw ex.getTargetException();
            }
            // 执行完成后需要切换回之前的撞他
            ContextHolder.setContext(context);
            DataSourceContextHolder.setDataSourceType(dataSource);
            return obj;
        }

    }
    public static String getDefaultDatabaseIdentifier() {

        return DBConfigInfoHelper.getDefaultDatabaseIdentifier();
    }
}
