package com.horizon.third;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 持久层接口
 * 
 * @author yufang
 *
 */
public interface Persistence {
	/**
	 * 获取数据表主键所需的32位主键值
	 * 
	 * @return 返回数据表主键所需的32位主键值
	 */
	public String getUUID();

	/**
	 * 批量更新数据.
	 * <p>
	 * 一个SQL语句对应一个条件值List对象
	 * 
	 * @param mapSqlParameters
	 *            一个以SQL语句为主键，以SQL语句对应的参数值列表对象为值的Map对象
	 * @return
	 */
	public boolean executeUpdate(Map<String, List<Object>> mapSqlParameters);

	/**
	 * 批量更新数据.
	 * <p>
	 * 一个SQL语句对应一个条件值List对象
	 * 
	 * <pre>
	 * mapSqlParameters参数数据结构：
	 * <div style="border: dashed 1px gray">
	 *     String sql1 = "delete from Xxxxx where id=?";
	 *     String sql2 = "insert into Xxxxx (id, name, ...) values (?,?, ...)";
	 * 
	 *     List conditions1 = new ArrayList(1);
	 *     List conditions2 = new ArrayList();
	 *     
	 *     conditions1.add(id);
	 *     
	 *     conditions2.add(newId);
	 *     conditions2.add(name);
	 *     ...
	 * 
	 *     LinkedHashMap mapSqlParameters = new LinkedHashMap(2);
	 *     mapSqlParameters.put(sql1, conditions1);
	 *     mapSqlParameters.put(sql2, conditions2);
	 * </div>
	 * </pre>
	 * 
	 * @param mapSqlParameters
	 *            一个以SQL语句为主键，以SQL语句对应的参数值列表对象为值的Map对象
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public boolean executeUpdate(Map<String, List<Object>> mapSqlParameters, String dbIdentifier);

	/**
	 * 以指定SQL语句和条件值进行数据更新
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstParameters
	 *            SQL语句的参数列表
	 * @return
	 */
	public boolean executeUpdate(String sql, List<? extends Object> lstParameters);

	/**
	 * 以指定SQL语句和条件值进行数据更新
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstParameters
	 *            所执行SQL语句对应的参数列表
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public boolean executeUpdate(String sql, List<? extends Object> lstParameters, String dbIdentifier);

	/**
	 * 批量更新多条数据.
	 * <p>
	 * 一个SQL语句对应多个条件值List对象
	 * 
	 * @param mapSqlParameters
	 *            一个以SQL语句为主键，以SQL语句对应的多个参数值列表对象为值的Map对象
	 * @return
	 */
	public boolean executeMultiUpdate(Map<String, List<Object>> mapSqlParameters);

	/**
	 * 批量更新多条数据.
	 * <p>
	 * 一个SQL语句对应多个条件值List对象
	 * 
	 * <pre>
	 * mapSqlParameters参数数据结构：
	 * <div style="border: dashed 1px gray">
	 *     String sql1 = "delete from Xxxxx where id=?";
	 *     String sql2 = "insert into Xxxxx (id, name, ...) values (?,?, ...)";
	 * 
	 *     List deleteConditions = new ArrayList();
	 *     List insertConditions = new ArrayList();
	 *     
	 *     List conditions1 = new ArrayList(1);
	 *     conditions1.add(id);
	 *     deleteConditions.add(conditions1);
	 *     ...;
	 *     
	 *     List conditions2 = new ArrayList();
	 *     conditions2.add(newId);
	 *     conditions2.add(name);
	 *     ...
	 *     insertConditions.add(conditions2);
	 *     ...;
	 * 
	 *     LinkedHashMap mapSqlParameters = new LinkedHashMap(2);
	 *     mapSqlParameters.put(sql1, deleteConditions);
	 *     mapSqlParameters.put(sql2, insertConditions);
	 * </div>
	 * </pre>
	 * 
	 * @param mapSqlParameters
	 *            一个以SQL语句为主键，以SQL语句对应的多个参数值列表对象为值的Map对象
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public boolean executeMultiUpdate(Map<String, ? extends Collection<?>> mapSqlParameters, String dbIdentifier);

	/**
	 * 保存CLOB
	 * 
	 * @param tableName
	 *            表名
	 * @param columnName
	 *            字段名称(CLOB)
	 * @param lstParameters
	 *            参数列表(依次包含两个参数值：ID值，clob字段值);
	 * @return
	 */
	public boolean saveClob(String tableName, String columnName, List<? extends Object> lstParameters);

	/**
	 * 保存CLOB
	 * 
	 * @param tableName
	 *            表名
	 * @param columnName
	 *            字段名称(CLOB)
	 * @param lstParameters
	 *            参数列表(依次包含两个参数值：ID值，clob字段值);
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public boolean saveClob(String tableName, String columnName, List<? extends Object> lstParameters,
			String dbIdentifier);

	/**
	 * 更新Blob字段
	 * 
	 * @param tableName
	 *            数据库表名
	 * @param columnName
	 *            字段名(BLOB)
	 * @param inputStream
	 *            输入流对象（对象中包含需要保存到Blob字段的内容）
	 * @param idValue
	 *            数据记录的主键值
	 * @return true-更新成功；false-更新失败
	 */
	public boolean updateBlob(String tableName, String columnName, InputStream inputStream, String idValue);

	/**
	 * 更新Blob字段
	 * 
	 * @param tableName
	 *            表名
	 * @param columnName
	 *            字段名(BLOB)
	 * @param inputStream
	 *            输入流对象（对象中包含需要保存到Blob字段的内容）
	 * @param idValue
	 *            数据记录的主键值
	 * @param dbIdentifier
	 *            数据库标识符
	 * @return true-更新成功；false-更新失败
	 */
	public boolean updateBlob(String tableName, String columnName, InputStream inputStream, String idValue,
			String dbIdentifier);

	/**
	 * 批量保存对象到数据库表中
	 * 
	 * @param mapTableNameAndDataObjects
	 *            需要保存的数据对象的集合（key指定的数据表名,value指定的数据表对应的数据对象）
	 * @return
	 */
	public boolean saveObjectData(Map<String, Object> mapTableNameAndDataObjects);

	/**
	 * 批量保存数据对象的值到数据库表中.
	 * 
	 * <pre>
	 * mapTableNameAndDataObjects参数数据结构：
	 * <div style="border: dashed 1px gray">
	 *     Context context = ContextFactory.getContext();
	 *     
	 *     Dept dept = (Dept) context.getDataObject(Dept.class);
	 *     User[] users = (User[]) context.getDataObjects(User.class);
	 *     
	 *     List lstDataObjects = new ArrayList();
	 *     lstDataObjects.add(obj1); // 已封装数据的数据对象
	 *     lstDataObjects.add(obj2); // 已封装数据的数据对象
	 *     ...
	 *     
	 *     LinkedHashMap mapTableNameAndDataObjects = new LinkedHashMap(2);
	 *     
	 *     // key: 表名; value: 数据对象 / 数据对象数组 / 包含数据对象的列表对象
	 *     mapTableNameAndDataObjects.put("TO_HORIZON_USER", users); // 数据对象数组
	 *     mapTableNameAndDataObjects.put("TO_HORIZON_dept", dept);  // 单个数据对象
	 *     mapTableNameAndDataObjects.put("data_table_name", lstDataObjects); // 包含数据对象的列表对象
	 * </div>
	 * </pre>
	 * 
	 * @param mapTableNameAndDataObjects
	 *            需要保存的数据对象的集合（key-为指定的数据表名; value-为指定的数据表对应的数据对象）
	 * @param dbIdentifier
	 *            数据库标识
	 * @return true-保存成功; false-保存失败
	 */
	public boolean saveObjectData(Map<String, Object> mapTableNameAndDataObjects, String dbIdentifier);

	/**
	 * 根据sql语句返回单条数据查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以List形式进行数据封装。<br/>
	 * 注意，如果查询返回多条数据，则此方法仅返回第一条数据。
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @return
	 */
	public List<Object> getSingleList(String sql, List<? extends Object> lstConditions);

	/**
	 * 根据sql语句返回单条数据查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以List形式进行数据封装。<br/>
	 * 注意，如果查询返回多条数据，则此方法仅返回第一条数据。
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dbIdentifier
	 *            数据库标识符
	 * @return
	 */
	public List<Object> getSingleList(String sql, List<? extends Object> lstConditions, String dbIdentifier);

	/**
	 * 根据sql语句返回多条数据查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以List形式进行数据封装
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @return
	 */
	public List<List<Object>> getMultiList(String sql, List<? extends Object> lstConditions);

	/**
	 * 根据sql语句返回多条数据查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以List形式进行数据封装
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public List<List<Object>> getMultiList(String sql, List<? extends Object> lstConditions, String dbIdentifier);

	/**
	 * 根据sql语句返回多条数据的分页查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以List形式进行数据封装。
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param iPageNum
	 *            页号
	 * @param iPageSize
	 *            每页记录数
	 * @return
	 */
	public List<List<Object>> getPagingList(String sql, List<? extends Object> lstConditions, int iPageNum,
			int iPageSize);

	/**
	 * 根据sql语句返回多条数据的分页查询结果的List.<br/>
	 * <p>
	 * 提示：单条数据以List形式进行数据封装。
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param iPageNum
	 *            页号
	 * @param iPageSize
	 *            每页记录数
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public List<List<Object>> getPagingList(String sql, List<? extends Object> lstConditions, int iPageNum,
			int iPageSize, String dbIdentifier);

	/**
	 * 根据sql语句返回单条数据查询结果的Map对象.<br/>
	 * <p>
	 * 提示：单条数据以Map形式进行数据封装。<br/>
	 * Map对象的key值为小写的字段名称,value为字段对应的值
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @return
	 */
	public Map<String, Object> getSingleMap(String sql, List<? extends Object> lstConditions);

	/**
	 * 根据sql语句返回单条数据查询结果的Map对象.<br/>
	 * <p>
	 * 提示：单条数据以Map形式进行数据封装。<br/>
	 * Map对象的key值为小写的字段名称,value为字段对应的值
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public Map<String, Object> getSingleMap(String sql, List<? extends Object> lstConditions, String dbIdentifier);

	/**
	 * 根据sql语句返回多条数据查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以Map形式进行数据封装。<br/>
	 * Map对象的key值为小写的字段名称,value为字段对应的值
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @return
	 */
	public List<Map<String, Object>> getMultiMap(String sql, List<? extends Object> lstConditions);

	/**
	 * 根据sql语句返回多条数据查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以Map形式进行数据封装。<br/>
	 * Map对象的key值为小写的字段名称,value为字段对应的值
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public List<Map<String, Object>> getMultiMap(String sql, List<? extends Object> lstConditions, String dbIdentifier);

	/**
	 * 根据sql语句返回多条数据的分页查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以Map形式进行数据封装。<br/>
	 * Map对象的key值为小写的字段名称,value为字段对应的值
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param iPageNum
	 *            页号
	 * @param iPageSize
	 *            每页记录数
	 * 
	 * @return
	 */
	public List<Map<String, Object>> getPagingMap(String sql, List<? extends Object> lstConditions, int iPageNum,
			int iPageSize);

	/**
	 * 根据sql语句返回多条数据的分页查询结果的List列表对象.<br/>
	 * <p>
	 * 提示：单条数据以Map形式进行数据封装。<br/>
	 * Map对象的key值为小写的字段名称,value为字段对应的值
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param iPageNum
	 *            页号
	 * @param iPageSize
	 *            每页记录数
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public List<Map<String, Object>> getPagingMap(String sql, List<? extends Object> lstConditions, int iPageNum,
			int iPageSize, String dbIdentifier);

	/**
	 * 根据sql语句返回单条数据查询结果的Object.<br/>
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dataObjectClass
	 *            自定义数据对象Class
	 * @return
	 */
	public <T> T getSingleObject(String sql, List<? extends Object> lstConditions, Class<T> dataObjectClass);

	/**
	 * 根据sql语句返回单条数据查询结果的Object.<br/>
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dataObjectClass
	 *            自定义数据对象Class
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public <T> T getSingleObject(String sql, List<? extends Object> lstConditions, Class<T> dataObjectClass,
			String dbIdentifier);

	/**
	 * 根据sql语句返回多条数据查询结果的List列表对象.<br/>
	 * <P>
	 * 提示：单条数据以自定义数据对象形式进行数据封装。<br/>
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dataObjectClass
	 *            自定义数据对象Class
	 * @return
	 */
	public <T> List<T> getMultiObject(String sql, List<? extends Object> lstConditions, Class<T> dataObjectClass);

	/**
	 * 根据sql语句返回多条数据查询结果的List列表对象.<br/>
	 * <P>
	 * 提示：单条数据以自定义数据对象形式进行数据封装。<br/>
	 * 
	 * <pre>
	 * 示例：
	 * <div style="border: dashed 1px gray">
	 *     String sql = "select * from Xxxxx where dept_id=?";
	 *     List lstConditions = new ArrayList(1);
	 *     
	 *     lstConditions.add(deptId);
	 *     
	 *     List lstUsers = Access.getMultiObject(sql, lstConditions, User.class, dbIdentifier);
	 * </div>
	 * </pre>
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dataObjectClass
	 *            自定义数据对象Class
	 * @param dbIdentifier
	 *            数据库标识
	 * @return 返回多条数据查询结果的List列表对象
	 */
	public <T> List<T> getMultiObject(String sql, List<? extends Object> lstConditions, Class<T> dataObjectClass,
			String dbIdentifier);

	/**
	 * 根据sql语句返回多条数据的分页查询结果的List列表对象.<br/>
	 * <P>
	 * 提示：单条数据以自定义数据对象形式进行数据封装。<br/>
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param iPageNum
	 *            页号
	 * @param iPageSize
	 *            每页记录数
	 * @param dataObjectClass
	 *            自定义数据对象Class
	 * @return
	 */
	public <T> List<T> getPagingObject(String sql, List<? extends Object> lstConditions, int iPageNum, int iPageSize,
			Class<T> dataObjectClass);

	/**
	 * 根据sql语句返回多条数据的分页查询结果的List列表对象.<br/>
	 * <P>
	 * 提示：单条数据以自定义数据对象形式进行数据封装。<br/>
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param iPageNum
	 *            页号
	 * @param iPageSize
	 *            每页记录数
	 * @param dataObjectClass
	 *            自定义数据对象Class
	 * @param dbIdentifier
	 *            数据库标识
	 * @return
	 */
	public <T> List<T> getPagingObject(String sql, List<? extends Object> lstConditions, int iPageNum, int iPageSize,
			Class<T> dataObjectClass, String dbIdentifier);

	/**
	 * 根据sql语句返回记录总数
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @return 返回记录总数
	 */
	public int getRsCount(String sql, List<? extends Object> lstConditions);

	/**
	 * 根据sql语句返回记录总数
	 * 
	 * @param sql
	 *            需要执行的SQL语句
	 * @param lstConditions
	 *            所执行SQL语句对应的参数列表
	 * @param dbIdentifier
	 *            数据库标识
	 * @return 返回记录总数
	 */
	public int getRsCount(String sql, List<? extends Object> lstConditions, String dbIdentifier);

	/**
	 * 获得CLOB内容.<br/>
	 * 获得默认数据库指定表中指定表名和字段、符合条件表达式的对应记录的Clob字段的值（转换为String类型）
	 * 
	 * @param tableName
	 *            表名
	 * @param fieldName
	 *            (CLOB字段的)字段名称
	 * @param condition
	 *            条件表达式
	 * @return CLOB内容字符串
	 */
	public String getClobContent(String tableName, String fieldName, String condition);

	/**
	 * 获得CLOB内容.<br/>
	 * 获得指定数据库标识对应数据库中指定表名和字段、符合条件表达式的对应记录的Clob字段的值（转换为String类型）
	 * 
	 * @param tableName
	 *            表名
	 * @param fieldName
	 *            (CLOB字段的)字段名称
	 * @param condition
	 *            条件表达式
	 * @param dbIdentifier
	 *            指定的数据库标识
	 * @return CLOB内容字符串
	 */
	public String getClobContent(String tableName, String fieldName, String condition, String dbIdentifier);

	/**
	 * （以字节数组形式）获得Blob字段的内容.<br/>
	 * 获得默认数据库标识对应数据库中指定表名和字段、符合条件表达式的对应记录的Blob字段的值
	 * 
	 * @param tableName
	 *            表名
	 * @param fieldName
	 *            (BLOB字段的)字段名称
	 * @param condition
	 *            条件表达式
	 * @return 返回Blob字段值的byte数组
	 */
	public byte[] getBlobContentBytes(String tableName, String fieldName, String condition);

	/**
	 * （以字节数组形式）获得Blob字段的内容.<br/>
	 * 获得指定数据库标识对应数据库中指定表名和字段、符合条件表达式的对应记录的Blob字段的值
	 * 
	 * @param tableName
	 *            表名
	 * @param fieldName
	 *            (BLOB字段的)字段名称
	 * @param condition
	 *            条件表达式
	 * @param dbIdentifier
	 *            指定的数据库标识
	 * @return 返回Blob字段值的byte数组
	 */
	public byte[] getBlobContentBytes(String tableName, String fieldName, String condition, String dbIdentifier);

	/**
	 * 获得默认数据库中指定数据表的字段信息
	 * 
	 * @param tableName
	 *            表名
	 * @return 返回默认数据库中指定数据表的字段信息 （Key: 小写字段名; Value:
	 *         com.horizon.module.TableColumn对象）
	 */
	public Map<String, Object> getTableColumnMsg(String tableName);

	/**
	 * 获得指定的数据表的字段信息
	 * 
	 * @param tableName
	 *            表名
	 * @param isToLowerCase
	 *            字段名大小写转换(true-小写；false-大写)
	 * @return 返回默认数据库中指定数据表的字段信息 （Key: 字段名; Value:
	 *         com.horizon.module.TableColumn对象）
	 */
	public Map<String, Object> getTableColumnMsg(String tableName, boolean isToLowerCase);

	/**
	 * 获得指定的数据表的字段信息
	 * 
	 * @param tableName
	 *            表名
	 * @param isToLowerCase
	 *            字段名大小写转换(true-小写；false-大写)
	 * @param dbIdentifier
	 *            数据库标识
	 * @return 返回默认数据库中指定数据表的字段信息 （Key: 字段名; Value:
	 *         com.horizon.module.TableColumn对象）
	 */
	public Map<String, Object> getTableColumnMsg(String tableName, boolean isToLowerCase, String dbIdentifier);

	/**
	 * 获得（默认数据库下）包含数据库表名的列表对象
	 * 
	 * @return 返回（默认数据库下）包含数据库表名的列表对象
	 */
	public List<Object> getTableNames();

	/**
	 * 获得（指定数据库下）包含数据库表名的列表对象
	 * 
	 * @param dbIdentifier
	 *            数据库标识符
	 * @return 返回（指定数据库下）包含数据库表名的列表对象
	 */
	public List<Object> getTableNames(String dbIdentifier);

	/**
	 * 获得指定数据源的数据库类型
	 * 
	 * @param dbIdentifier
	 * @return ORACLE,SQLSERVER,DB2,MYSQL,DM
	 */
	public String getDbType(String dbIdentifier);

	/**
	 * 获得默认数据源的数据库类型
	 * 
	 * @return ORACLE,SQLSERVER,DB2,MYSQL,DM
	 */
	public String getDefaultDbType();

	/**
	 * 数据库链接标识有效性判断
	 * 
	 * @param dbIdentifier
	 * @return true :有效 false:无效
	 */
	public boolean validateIdentifier(String dbIdentifier);

	/**
	 * 获得默认的数据库标识
	 * 
	 * @return
	 */
	public String getDefaultDatabaseIdentifier();

}
