/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  [2002] - [2007] Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package flex.messaging.config;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

import flex.messaging.util.LocaleUtils;

/**
 * Processes DOM representation of a messaging configuration file.
 * <p>
 * Note: Since reference ids are used between elements, certain
 * sections of the document need to be parsed first.
 * </p>
 *
 * @author Peter Farland
 * @exclude
 */
public abstract class ServerConfigurationParser extends AbstractConfigurationParser
{
    protected void parseTopLevelConfig(Document doc)
    {
        Node root = selectSingleNode(doc, "/" + SERVICES_CONFIG_ELEMENT);

        if (root != null)
        {
            // Validation
            allowedChildElements(root, SERVICES_CONFIG_CHILDREN);

            // Security (parse before channels, services)
            securitySection(root);

            // Shared servers
            serversSection(root);

            // Channels (parse before services)
            channelsSection(root);

            // Services
            services(root);

            // Clustering
            clusters(root);

            // Logging
            logging(root);

            // System
            system(root);

            // FlexClient
            flexClient(root);

            // The factories
            factories(root);
        }
        else
        {
            // The services configuration root element must be '{SERVICES_CONFIG_ELEMENT}'.
            ConfigurationException e = new ConfigurationException();
            e.setMessage(INVALID_SERVICES_ROOT, new Object[]{SERVICES_CONFIG_ELEMENT});
            throw e;
        }
    }

    private void clusters(Node root)
    {
        Node clusteringNode = selectSingleNode(root, CLUSTERS_ELEMENT);
        if (clusteringNode != null)
        {
            allowedAttributesOrElements(clusteringNode, CLUSTERING_CHILDREN);

            NodeList clusters = selectNodeList(clusteringNode, CLUSTER_DEFINITION_ELEMENT);
            for (int i = 0; i < clusters.getLength(); i++)
            {
                Node cluster = clusters.item(i);
                requiredAttributesOrElements(cluster, CLUSTER_DEFINITION_CHILDREN);
                String clusterName = getAttributeOrChildElement(cluster, ID_ATTR);
                if (isValidID(clusterName))
                {
                    String propsFileName = getAttributeOrChildElement(cluster, CLUSTER_PROPERTIES_ATTR);
                    ClusterSettings clusterSettings = new ClusterSettings();
                    clusterSettings.setClusterName(clusterName);
                    clusterSettings.setPropsFileName(propsFileName);
                    String className = getAttributeOrChildElement(cluster, CLASS_ATTR);
                    if (className != null && className.length() > 0)
                    {
                        clusterSettings.setImplementationClass(className);
                    }
                    String defaultValue = getAttributeOrChildElement(cluster, ClusterSettings.DEFAULT_ELEMENT);
                    if (defaultValue != null && defaultValue.length() > 0)
                    {
                        if (defaultValue.equalsIgnoreCase("true"))
                            clusterSettings.setDefault(true);
                        else if (!defaultValue.equalsIgnoreCase("false"))
                        {
                            ConfigurationException e = new ConfigurationException();
                            e.setMessage(10215, new Object[] {clusterName, defaultValue});
                            throw e;
                        }
                    }
                    String ulb = getAttributeOrChildElement(cluster, ClusterSettings.URL_LOAD_BALANCING);
                    if (ulb != null && ulb.length() > 0)
                    {
                        if (ulb.equalsIgnoreCase("false"))
                            clusterSettings.setURLLoadBalancing(false);
                        else if (!ulb.equalsIgnoreCase("true"))
                        {
                            ConfigurationException e = new ConfigurationException();
                            e.setMessage(10216, new Object[] {clusterName, ulb});
                            throw e;
                        }
                    }
                    ((MessagingConfiguration)config).addClusterSettings(clusterSettings);
                }
            }
        }
    }


    private void securitySection(Node root)
    {
        Node security = selectSingleNode(root, SECURITY_ELEMENT);

        if (security != null)
        {
            // Validation
            allowedChildElements(security, SECURITY_CHILDREN);

            // Security Constraints
            NodeList list = selectNodeList(security, SECURITY_CONSTRAINT_DEFINITION_ELEMENT);
            for (int i = 0; i < list.getLength(); i++)
            {
                Node constraint = list.item(i);
                securityConstraint(constraint, false);
            }

            // Login Commands
            list = selectNodeList(security, LOGIN_COMMAND_ELEMENT);
            for (int i = 0; i < list.getLength(); i++)
            {
                Node login = list.item(i);
                LoginCommandSettings loginCommandSettings= new LoginCommandSettings();
                requiredAttributesOrElements(login, LOGIN_COMMAND_REQ_CHILDREN);
                allowedAttributesOrElements(login, LOGIN_COMMAND_CHILDREN);

                String server = getAttributeOrChildElement(login, SERVER_ATTR);
                if (server.length() == 0)
                {
                    // Attribute '{SERVER_ATTR}' must be specified for element '{LOGIN_COMMAND_ELEMENT}'
                    ConfigurationException e = new ConfigurationException();
                    e.setMessage(MISSING_ATTRIBUTE, new Object[]{SERVER_ATTR, LOGIN_COMMAND_ELEMENT});
                    throw e;
                }
                loginCommandSettings.setServer(server);

                String loginClass = getAttributeOrChildElement(login, CLASS_ATTR);
                if (loginClass.length() == 0)
                {
                    // Attribute '{CLASS_ATTR}' must be specified for element '{LOGIN_COMMAND_ELEMENT}'
                    ConfigurationException e = new ConfigurationException();
                    e.setMessage(MISSING_ATTRIBUTE, new Object[]{CLASS_ATTR, LOGIN_COMMAND_ELEMENT});
                    throw e;
                }
                loginCommandSettings.setClassName(loginClass);

                boolean isPerClientAuth = Boolean.valueOf(getAttributeOrChildElement(login, PER_CLIENT_AUTH)).booleanValue();
                loginCommandSettings.setPerClientAuthentication(isPerClientAuth);

                ((MessagingConfiguration)config).getSecuritySettings().addLoginCommandSettings(loginCommandSettings);
            }
        }
    }

    private SecurityConstraint securityConstraint(Node constraint, boolean inline)
    {
        SecurityConstraint sc;

        // Validation
        allowedAttributesOrElements(constraint, SECURITY_CONSTRAINT_DEFINITION_CHILDREN);

        // Constraint by reference
        String ref = getAttributeOrChildElement(constraint, REF_ATTR);
        if (ref.length() > 0)
        {
            allowedAttributesOrElements(constraint, new String[] {REF_ATTR});

            sc = ((MessagingConfiguration)config).getSecuritySettings().getConstraint(ref);
            if (sc == null)
            {
                // {SECURITY_CONSTRAINT_DEFINITION_ELEMENT} not found for reference '{ref}'.
                ConfigurationException e = new ConfigurationException();
                e.setMessage(REF_NOT_FOUND, new Object[]{SECURITY_CONSTRAINT_DEFINITION_ELEMENT, ref});
                throw e;
            }
        }
        else
        {
            // New security constraint
            String id = getAttributeOrChildElement(constraint, ID_ATTR);

            // If not inline, we must have a valid id to register the constraint!
            if (inline)
            {
                sc = new SecurityConstraint("");
            }
            else if (isValidID(id))
            {
                sc = new SecurityConstraint(id);
                ((MessagingConfiguration)config).getSecuritySettings().addConstraint(sc);
            }
            else
            {
                //Invalid {SECURITY_CONSTRAINT_DEFINITION_ELEMENT} id '{id}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(INVALID_ID, new Object[]{SECURITY_CONSTRAINT_DEFINITION_ELEMENT, id});
                ex.setDetails(INVALID_ID);
                throw ex;
            }

            // Authentication Method
            String method = getAttributeOrChildElement(constraint, AUTH_METHOD_ELEMENT);
            sc.setMethod(method);

            // Roles
            Node rolesNode = selectSingleNode(constraint, ROLES_ELEMENT);
            if (rolesNode != null)
            {
                allowedChildElements(rolesNode, ROLES_CHILDREN);
                NodeList roles = selectNodeList(rolesNode, ROLE_ELEMENT);
                for (int r = 0; r < roles.getLength(); r++)
                {
                    Node roleNode = roles.item(r);
                    String role = evaluateExpression(roleNode, ".").toString().trim();
                    if (role.length() > 0)
                    {
                        sc.addRole(role);
                    }
                }
            }
        }

        return sc;
    }

    private void serversSection(Node root)
    {
        // Only MessagingConfiguration supports the servers element configuration.
        // The general ServicesConfiguration interface does not.
        if (!(config instanceof MessagingConfiguration))
            return;

        Node serversNode = selectSingleNode(root, SERVERS_ELEMENT);
        if (serversNode != null)
        {
            // Validation
            allowedAttributesOrElements(serversNode, SERVERS_CHILDREN);

            NodeList servers = selectNodeList(serversNode, SERVER_ELEMENT);
            for (int i = 0; i < servers.getLength(); i++)
            {
                Node server = servers.item(i);
                serverDefinition(server);
            }
        }
    }

    private void serverDefinition(Node server)
    {
        // Validation
        requiredAttributesOrElements(server, SERVER_REQ_CHILDREN);

        String id = getAttributeOrChildElement(server, ID_ATTR);
        if (isValidID(id))
        {
            SharedServerSettings settings = new SharedServerSettings();
            settings.setId(id);
            settings.setSourceFile(getSourceFileOf(server));
            settings.setClassName(getAttributeOrChildElement(server, CLASS_ATTR));
            // Custom server properties.
            NodeList properties = selectNodeList(server, PROPERTIES_ELEMENT + "/*");
            if (properties.getLength() > 0)
            {
                ConfigMap map = properties(properties, getSourceFileOf(server));
                settings.addProperties(map);
            }
            ((MessagingConfiguration)config).addSharedServerSettings(settings);
        }
    }

    private void channelsSection(Node root)
    {
        Node channelsNode = selectSingleNode(root, CHANNELS_ELEMENT);
        if (channelsNode != null)
        {
            // Validation
            allowedAttributesOrElements(channelsNode, CHANNELS_CHILDREN);

            NodeList channels = selectNodeList(channelsNode, CHANNEL_DEFINITION_ELEMENT);
            for (int i = 0; i < channels.getLength(); i++)
            {
                Node channel = channels.item(i);
                channelDefinition(channel);
            }
        }
    }

    private void channelDefinition(Node channel)
    {
        // Validation
        requiredAttributesOrElements(channel, CHANNEL_DEFINITION_REQ_CHILDREN);
        allowedAttributesOrElements(channel, CHANNEL_DEFINITION_CHILDREN);

        String id = getAttributeOrChildElement(channel, ID_ATTR);
        if (isValidID(id))
        {
            // Don't allow multiple channels with the same id
            if (config.getChannelSettings(id) != null)
            {
                // Cannot have multiple channels with the same id ''{0}''.
                ConfigurationException e = new ConfigurationException();
                e.setMessage(DUPLICATE_CHANNEL_ERROR, new Object[]{id});
                throw e;
            }

            ChannelSettings channelSettings = new ChannelSettings(id);
            channelSettings.setSourceFile(getSourceFileOf(channel));

            String clientType = getAttributeOrChildElement(channel, CLASS_ATTR);
            channelSettings.setClientType(clientType);

            // Note whether the channel-definition was for a remote endpoint
            String remote = getAttributeOrChildElement(channel, REMOTE_ATTR);
            channelSettings.setRemote(Boolean.valueOf(remote).booleanValue());

            // Endpoint
            Node endpoint = selectSingleNode(channel, ENDPOINT_ELEMENT);
            if (endpoint != null)
            {
                // Endpoint Validation
                allowedAttributesOrElements(endpoint, ENDPOINT_CHILDREN);

                String type = getAttributeOrChildElement(endpoint, CLASS_ATTR);
                channelSettings.setEndpointType(type);

                // The url attribute may also be specified by the deprecated uri attribute
                String uri = getAttributeOrChildElement(endpoint, URL_ATTR);
                if (uri == null || EMPTY_STRING.equals(uri))
                    uri = getAttributeOrChildElement(endpoint, URI_ATTR);
                channelSettings.setUri(uri);

                config.addChannelSettings(id, channelSettings);
            }

            // Server reference
            Node server = selectSingleNode(channel, SERVER_ELEMENT);
            if (server != null)
            {
                requiredAttributesOrElements(server, CHANNEL_DEFINITION_SERVER_REQ_CHILDREN);

                String serverId = getAttributeOrChildElement(server, REF_ATTR);
                channelSettings.setServerId(serverId);
            }

            // Channel Properties
            NodeList properties = selectNodeList(channel, PROPERTIES_ELEMENT + "/*");
            if (properties.getLength() > 0)
            {
                ConfigMap map = properties(properties, getSourceFileOf(channel));
                channelSettings.addProperties(map);
            }

            // Channel Security

            // Security-constraint short-cut attribute
            String ref = evaluateExpression(channel, "@" + SECURITY_CONSTRAINT_ATTR).toString().trim();
            if (ref.length() > 0)
            {
                SecurityConstraint sc = ((MessagingConfiguration)config).getSecuritySettings().getConstraint(ref);
                if (sc != null)
                {
                    channelSettings.setConstraint(sc);
                }
                else
                {
                    // {SECURITY_CONSTRAINT_ELEMENT} not found for reference '{ref}' in channel '{id}'.
                    ConfigurationException ex = new ConfigurationException();
                    ex.setMessage(REF_NOT_FOUND_IN_CHANNEL, new Object[]{SECURITY_CONSTRAINT_ATTR, ref, id});
                    throw ex;
                }
            }
            else
            {
                // Inline security element
                Node security = selectSingleNode(channel, SECURITY_ELEMENT);
                if (security != null)
                {
                    allowedChildElements(security, EMBEDDED_SECURITY_CHILDREN);
                    Node constraint = selectSingleNode(security, SECURITY_CONSTRAINT_ELEMENT);
                    if (constraint != null)
                    {
                        SecurityConstraint sc = securityConstraint(constraint, true);
                        channelSettings.setConstraint(sc);
                    }
                }
            }
        }
        else
        {
            // Invalid {CHANNEL_DEFINITION_ELEMENT} id '{id}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_ID, new Object[]{CHANNEL_DEFINITION_ELEMENT, id});
            ex.setDetails(INVALID_ID);
            throw ex;
        }
    }

    private void services(Node root)
    {
        Node servicesNode = selectSingleNode(root, SERVICES_ELEMENT);
        if (servicesNode != null)
        {
            // Validation
            allowedChildElements(servicesNode, SERVICES_CHILDREN);

            // Default Channels for the application
            Node defaultChannels = selectSingleNode(servicesNode, DEFAULT_CHANNELS_ELEMENT);
            if (defaultChannels != null)
            {
                allowedChildElements(defaultChannels, DEFAULT_CHANNELS_CHILDREN);
                NodeList channels = selectNodeList(defaultChannels, CHANNEL_ELEMENT);
                for (int c = 0; c < channels.getLength(); c++)
                {
                    Node chan = channels.item(c);
                    allowedAttributes(chan, new String[] {REF_ATTR});
                    defaultChannel(chan);
                }
            }

            // Service Includes
            NodeList services = selectNodeList(servicesNode, SERVICE_INCLUDE_ELEMENT);
            for (int i = 0; i < services.getLength(); i++)
            {
                Node service = services.item(i);
                serviceInclude(service);
            }

            // Service
            services = selectNodeList(servicesNode, SERVICE_ELEMENT);
            for (int i = 0; i < services.getLength(); i++)
            {
                Node service = services.item(i);
                service(service);
            }
        }
    }

    private void serviceInclude(Node serviceInclude)
    {
        // Validation
        requiredAttributesOrElements(serviceInclude, SERVICE_INCLUDE_CHILDREN);

        String src = getAttributeOrChildElement(serviceInclude, SRC_ATTR);
        if (src.length() > 0)
        {
            Document doc = loadDocument(src, fileResolver.getIncludedFile(src));
            doc.getDocumentElement().normalize();

            Node service = selectSingleNode(doc, "/" + SERVICE_ELEMENT);
            if (service != null)
            {
                service(service);
                fileResolver.popIncludedFile();
            }
            else
            {
                // The services include root element must be '{SERVICE_ELEMENT}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(INVALID_SERVICE_INCLUDE_ROOT, new Object[]{SERVICE_ELEMENT});
                throw ex;
            }
        }
    }

    private void service(Node service)
    {
        // Validation
        requiredAttributesOrElements(service, SERVICE_REQ_CHILDREN);
        allowedAttributesOrElements(service, SERVICE_CHILDREN);

        String id = getAttributeOrChildElement(service, ID_ATTR);
        if (isValidID(id))
        {
            ServiceSettings serviceSettings = config.getServiceSettings(id);
            if (serviceSettings == null)
            {
                serviceSettings = new ServiceSettings(id);
                serviceSettings.setSourceFile(getSourceFileOf(service));
                config.addServiceSettings(serviceSettings);
            }
            else
            {
                // Duplicate service definition '{0}'.
                ConfigurationException e = new ConfigurationException();
                e.setMessage(DUPLICATE_SERVICE_ERROR, new Object[]{id});
                throw e;
            }

            // Service Class Name
            String className = getAttributeOrChildElement(service, CLASS_ATTR);
            if (className.length() > 0)
            {
                serviceSettings.setClassName(className);
            }
            else
            {
                // Class not specified for {SERVICE_ELEMENT} '{id}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(CLASS_NOT_SPECIFIED, new Object[]{SERVICE_ELEMENT, id});
                throw ex;
            }

            // Service Message Types - deprecated

            // Service Properties
            NodeList properties = selectNodeList(service, PROPERTIES_ELEMENT + "/*");
            if (properties.getLength() > 0)
            {
                ConfigMap map = properties(properties, getSourceFileOf(service));
                serviceSettings.addProperties(map);
            }

            // Default Channels
            Node defaultChannels = selectSingleNode(service, DEFAULT_CHANNELS_ELEMENT);
            if (defaultChannels != null)
            {
                allowedChildElements(defaultChannels, DEFAULT_CHANNELS_CHILDREN);
                NodeList channels = selectNodeList(defaultChannels, CHANNEL_ELEMENT);
                for (int c = 0; c < channels.getLength(); c++)
                {
                    Node chan = channels.item(c);
                    allowedAttributes(chan, new String[] {REF_ATTR});
                    defaultChannel(chan, serviceSettings);
                }
            }
            // Fall back on application's default channels
            else if (config.getDefaultChannels().size() > 0)
            {
                for (Iterator iter = config.getDefaultChannels().iterator(); iter.hasNext();)
                {
                    String channelId = (String)iter.next();
                    ChannelSettings channel = config.getChannelSettings(channelId);
                    serviceSettings.addDefaultChannel(channel);
                }
            }

            // Default Security Constraint
            Node defaultSecurityConstraint = selectSingleNode(service, DEFAULT_SECURITY_CONSTRAINT_ELEMENT);
            if (defaultSecurityConstraint != null)
            {
                // Validation
                requiredAttributesOrElements(defaultSecurityConstraint, new String[] {REF_ATTR});
                allowedAttributesOrElements(defaultSecurityConstraint, new String[] {REF_ATTR});

                String ref = getAttributeOrChildElement(defaultSecurityConstraint, REF_ATTR);
                if (ref.length() > 0)
                {
                    SecurityConstraint sc = ((MessagingConfiguration)config).getSecuritySettings().getConstraint(ref);
                    if (sc == null)
                    {
                        // {SECURITY_CONSTRAINT_DEFINITION_ELEMENT} not found for reference '{ref}'.
                        ConfigurationException e = new ConfigurationException();
                        e.setMessage(REF_NOT_FOUND, new Object[]{SECURITY_CONSTRAINT_DEFINITION_ELEMENT, ref});
                        throw e;
                    }
                    serviceSettings.setConstraint(sc);
                }
                else
                {
                    //Invalid default-security-constraint reference ''{0}'' in service ''{1}''.
                    ConfigurationException ex = new ConfigurationException();
                    ex.setMessage(INVALID_SECURITY_CONSTRAINT_REF, new Object[]{ref, id});
                    throw ex;
                }
            }

            // Adapter Definitions
            Node adapters = selectSingleNode(service, ADAPTERS_ELEMENT);
            if (adapters != null)
            {
                allowedChildElements(adapters, ADAPTERS_CHILDREN);
                NodeList serverAdapters = selectNodeList(adapters, ADAPTER_DEFINITION_ELEMENT);
                for (int a = 0; a < serverAdapters.getLength(); a++)
                {
                    Node adapter = serverAdapters.item(a);
                    adapterDefinition(adapter, serviceSettings);
                }
            }

            // Destinations
            NodeList list = selectNodeList(service, DESTINATION_ELEMENT);
            for (int i = 0; i < list.getLength(); i++)
            {
                Node dest = list.item(i);
                destination(dest, serviceSettings);
            }

            // Destination Includes
            list = selectNodeList(service, DESTINATION_INCLUDE_ELEMENT);
            for (int i = 0; i < list.getLength(); i++)
            {
                Node dest = list.item(i);
                destinationInclude(dest, serviceSettings);
            }
        }
        else
        {
            //Invalid {SERVICE_ELEMENT} id '{id}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_ID, new Object[]{SERVICE_ELEMENT, id});
            throw ex;
        }
    }

    /**
     * A Flex application can declare default channels for its services. If a
     * service specifies its own list of channels it overrides these defaults.
     * <p>
     * &lt;default-channels&gt;<br />
     * &nbsp;&nbsp;&nbsp;&nbsp;&lt;channel ref="channel-id" /&gt;<br />
     * &lt;default-channels&gt;
     * </p>
     */
    private void defaultChannel(Node chan)
    {
        String ref = getAttributeOrChildElement(chan, REF_ATTR);

        if (ref.length() > 0)
        {
            ChannelSettings channel = config.getChannelSettings(ref);
            if (channel != null)
            {
                config.addDefaultChannel(channel.getId());
            }
            else
            {
                // {0} not found for reference '{1}'
                ConfigurationException e = new ConfigurationException();
                e.setMessage(REF_NOT_FOUND, new Object[]{CHANNEL_ELEMENT, ref});
                throw e;
            }
        }
        else
        {
            //A default channel was specified without a reference for service '{0}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_DEFAULT_CHANNEL, new Object[]{"MessageBroker"});
            throw ex;
        }
    }

    /**
     * A service can declare default channels for its destinations. If a destination
     * specifies its own list of channels it overrides these defaults.
     * <p>
     * &lt;default-channels&gt;<br />
     * &nbsp;&nbsp;&nbsp;&nbsp;&lt;channel ref="channel-id" /&gt;<br />
     * &lt;default-channels&gt;
     * </p>
     */
    private void defaultChannel(Node chan, ServiceSettings serviceSettings)
    {
        String ref = getAttributeOrChildElement(chan, REF_ATTR);

        if (ref.length() > 0)
        {
            ChannelSettings channel = config.getChannelSettings(ref);
            if (channel != null)
            {
                serviceSettings.addDefaultChannel(channel);
            }
            else
            {
                // {0} not found for reference '{1}'
                ConfigurationException e = new ConfigurationException();
                e.setMessage(REF_NOT_FOUND, new Object[]{CHANNEL_ELEMENT, ref});
                throw e;
            }
        }
        else
        {
            //A default channel was specified without a reference for service '{0}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_DEFAULT_CHANNEL, new Object[]{serviceSettings.getId()});
            throw ex;
        }
    }

    private void adapterDefinition(Node adapter, ServiceSettings serviceSettings)
    {
        // Validation
        requiredAttributesOrElements(adapter, ADAPTER_DEFINITION_REQ_CHILDREN);
        allowedChildElements(adapter, ADAPTER_DEFINITION_CHILDREN);

        String serviceId = serviceSettings.getId();

        String id = getAttributeOrChildElement(adapter, ID_ATTR);
        if (isValidID(id))
        {
            AdapterSettings adapterSettings = new AdapterSettings(id);
            adapterSettings.setSourceFile(getSourceFileOf(adapter));
            String className = getAttributeOrChildElement(adapter, CLASS_ATTR);

            if (className.length() > 0)
            {
                adapterSettings.setClassName(className);

                // Default Adapter Check
                boolean isDefault = Boolean.valueOf(getAttributeOrChildElement(adapter, DEFAULT_ATTR)).booleanValue();
                if (isDefault)
                {
                    adapterSettings.setDefault(isDefault);

                    AdapterSettings defaultAdapter;
                    defaultAdapter = serviceSettings.getDefaultAdapter();

                    if (defaultAdapter != null)
                    {
                        // Duplicate default adapter '{0}' in service '{1}'. '{2}' has already been selected as the default.
                        ConfigurationException ex = new ConfigurationException();
                        ex.setMessage(DUPLICATE_DEFAULT_ADAPTER, new Object[]{id, serviceId, defaultAdapter.getId()});
                        throw ex;
                    }
                }

                serviceSettings.addAdapterSettings(adapterSettings);

                // Adapter Properties
                NodeList properties = selectNodeList(adapter, PROPERTIES_ELEMENT + "/*");
                if (properties.getLength() > 0)
                {
                    ConfigMap map = properties(properties, getSourceFileOf(adapter));
                    adapterSettings.addProperties(map);
                }
            }
            else
            {
                // Class not specified for {ADAPTER_DEFINITION_ELEMENT} '{id}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(CLASS_NOT_SPECIFIED, new Object[]{ADAPTER_DEFINITION_ELEMENT, id});
                throw ex;
            }
        }
        else
        {
            //Invalid {ADAPTER_DEFINITION_ELEMENT} id '{id}' for service '{serviceId}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_ID_IN_SERVICE, new Object[]{ADAPTER_DEFINITION_ELEMENT, id, serviceId});
            throw ex;
        }
    }

    private void destinationInclude(Node destInclude, ServiceSettings serviceSettings)
    {
        // Validation
        requiredAttributesOrElements(destInclude, DESTINATION_INCLUDE_CHILDREN);

        String src = getAttributeOrChildElement(destInclude, SRC_ATTR);
        if (src.length() > 0)
        {
            Document doc = loadDocument(src, fileResolver.getIncludedFile(src));
            doc.getDocumentElement().normalize();

            Node dest = selectSingleNode(doc, "/" + DESTINATION_ELEMENT);
            if (dest != null)
            {
                destination(dest, serviceSettings);
                fileResolver.popIncludedFile();
            }
            else
            {
                //The destination include root element must be '{DESTINATION_ELEMENT}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(INVALID_DESTINATION_INCLUDE_ROOT, new Object[]{DESTINATION_ELEMENT});
                throw ex;
            }
        }
    }

    private void destination(Node dest, ServiceSettings serviceSettings)
    {
        // Validation
        requiredAttributesOrElements(dest, DESTINATION_REQ_CHILDREN);
        allowedAttributes(dest, DESTINATION_ATTR);
        allowedChildElements(dest, DESTINATION_CHILDREN);

        String serviceId = serviceSettings.getId();

        DestinationSettings destinationSettings;
        String id = getAttributeOrChildElement(dest, ID_ATTR);
        if (isValidID(id))
        {
            destinationSettings = (DestinationSettings)serviceSettings.getDestinationSettings().get(id);
            if (destinationSettings != null)
            {
                // Duplicate destination definition '{id}' in service '{serviceId}'.
                ConfigurationException e = new ConfigurationException();
                e.setMessage(DUPLICATE_DESTINATION_ERROR, new Object[]{id, serviceId});
                throw e;
            }

            destinationSettings = new DestinationSettings(id);
            destinationSettings.setSourceFile(getSourceFileOf(dest));
            serviceSettings.addDestinationSettings(destinationSettings);
        }
        else
        {
            //Invalid {DESTINATION_ELEMENT} id '{id}' for service '{serviceId}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_ID_IN_SERVICE, new Object[]{DESTINATION_ELEMENT, id, serviceId});
            throw ex;
        }

        // Destination Properties
        NodeList properties = selectNodeList(dest, PROPERTIES_ELEMENT + "/*");
        if (properties.getLength() > 0)
        {
            ConfigMap map = properties(properties, getSourceFileOf(dest));
            destinationSettings.addProperties(map);
        }

        // Channels
        destinationChannels(dest, destinationSettings, serviceSettings);

        // Security
        destinationSecurity(dest, destinationSettings, serviceSettings);

        // Service Adapter
        destinationAdapter(dest, destinationSettings, serviceSettings);
    }

    private void destinationChannels(Node dest, DestinationSettings destinationSettings, ServiceSettings serviceSettings)
    {
        String destId = destinationSettings.getId();

        // Channels attribute
        String channelsList = evaluateExpression(dest, "@" + CHANNELS_ATTR).toString().trim();
        if (channelsList.length() > 0)
        {
            StringTokenizer st = new StringTokenizer(channelsList, LIST_DELIMITERS);
            while (st.hasMoreTokens())
            {
                String ref = st.nextToken().trim();
                ChannelSettings channel = config.getChannelSettings(ref);
                if (channel != null)
                {
                    destinationSettings.addChannelSettings(channel);
                }
                else
                {
                    // {CHANNEL_ELEMENT} not found for reference '{ref}' in destination '{destId}'.
                    ConfigurationException ex = new ConfigurationException();
                    ex.setMessage(REF_NOT_FOUND_IN_DEST, new Object[]{CHANNEL_ELEMENT, ref, destId});
                    throw ex;
                }
            }
        }
        else
        {
            // Channels element
            Node channelsNode = selectSingleNode(dest, CHANNELS_ELEMENT);
            if (channelsNode != null)
            {
                allowedChildElements(channelsNode, DESTINATION_CHANNELS_CHILDREN);
                NodeList channels = selectNodeList(channelsNode, CHANNEL_ELEMENT);
                for (int c = 0; c < channels.getLength(); c++)
                {
                    Node chan = channels.item(c);

                    // Validation
                    requiredAttributesOrElements(chan, DESTINATION_CHANNEL_REQ_CHILDREN);

                    String ref = getAttributeOrChildElement(chan, REF_ATTR);
                    if (ref.length() > 0)
                    {
                        ChannelSettings channel = config.getChannelSettings(ref);
                        if (channel != null)
                        {
                            destinationSettings.addChannelSettings(channel);
                        }
                        else
                        {
                            // {CHANNEL_ELEMENT} not found for reference '{ref}' in destination '{destId}'.
                            ConfigurationException ex = new ConfigurationException();
                            ex.setMessage(REF_NOT_FOUND_IN_DEST, new Object[]{CHANNEL_ELEMENT, ref, destId});
                            throw ex;
                        }
                    }
                    else
                    {
                        //Invalid {0} ref '{1}' in destination '{2}'.
                        ConfigurationException ex = new ConfigurationException();
                        ex.setMessage(INVALID_REF_IN_DEST, new Object[]{CHANNEL_ELEMENT, ref, destId});
                        throw ex;
                    }
                }
            }
            else
            {
                // Finally, we fall back to the service's default channels
                List defaultChannels = serviceSettings.getDefaultChannels();
                Iterator it = defaultChannels.iterator();
                while (it.hasNext())
                {
                    ChannelSettings channel = (ChannelSettings)it.next();
                    destinationSettings.addChannelSettings(channel);
                }
            }
        }

        if (destinationSettings.getChannelSettings().size() <= 0)
        {
            // Destination '{id}' must specify at least one channel.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(DEST_NEEDS_CHANNEL, new Object[]{destId});
            throw ex;
        }
    }

    private void destinationSecurity(Node dest, DestinationSettings destinationSettings, ServiceSettings serviceSettings)
    {
        String destId = destinationSettings.getId();

        // Security-constraint short-cut attribute
        String ref = evaluateExpression(dest, "@" + SECURITY_CONSTRAINT_ATTR).toString().trim();
        if (ref.length() > 0)
        {
            SecurityConstraint sc = ((MessagingConfiguration)config).getSecuritySettings().getConstraint(ref);
            if (sc != null)
            {
                destinationSettings.setConstraint(sc);
            }
            else
            {
                // {SECURITY_CONSTRAINT_ELEMENT} not found for reference '{ref}' in destination '{destId}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(REF_NOT_FOUND_IN_DEST, new Object[]{SECURITY_CONSTRAINT_ATTR, ref, destId});
                throw ex;
            }
        }
        else
        {
            // Inline security element
            Node security = selectSingleNode(dest, SECURITY_ELEMENT);
            if (security != null)
            {
                allowedChildElements(security, EMBEDDED_SECURITY_CHILDREN);
                Node constraint = selectSingleNode(security, SECURITY_CONSTRAINT_ELEMENT);
                if (constraint != null)
                {
                    SecurityConstraint sc = securityConstraint(constraint, true);
                    destinationSettings.setConstraint(sc);
                }
            }
            else
            {
                // Finally, we fall back to the service's default security constraint
                SecurityConstraint sc = serviceSettings.getConstraint();
                if (sc != null)
                {
                    destinationSettings.setConstraint(sc);
                }
            }
        }
    }

    private void destinationAdapter(Node dest, DestinationSettings destinationSettings, ServiceSettings serviceSettings)
    {
        String destId = destinationSettings.getId();

        // Adapter attribute
        String ref = evaluateExpression(dest, "@" + ADAPTER_ATTR).toString().trim();
        if (ref.length() > 0)
        {
            adapterReference(ref, destinationSettings, serviceSettings);
        }
        else
        {
            Node adapter = selectSingleNode(dest, ADAPTER_ELEMENT);

            // Adapter element
            if (adapter != null)
            {
                allowedAttributesOrElements(adapter, DESTINATION_ADAPTER_CHILDREN);
                ref = getAttributeOrChildElement(adapter, REF_ATTR);
                adapterReference(ref, destinationSettings, serviceSettings);
            }
            else
            {
                // Default Adapter (optionally set at the service level)
                AdapterSettings adapterSettings = serviceSettings.getDefaultAdapter();
                if (adapterSettings != null)
                {
                    destinationSettings.setAdapterSettings(adapterSettings);
                }
            }
        }

        if (destinationSettings.getAdapterSettings() == null)
        {
            // Destination '{id}' must specify at least one adapter.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(DEST_NEEDS_ADAPTER, new Object[]{destId});
            throw ex;
        }
    }

    private void adapterReference(String ref, DestinationSettings destinationSettings, ServiceSettings serviceSettings)
    {
        String destId = destinationSettings.getId();
        if (ref.length() > 0)
        {
            AdapterSettings adapterSettings = serviceSettings.getAdapterSettings(ref);
            if (adapterSettings != null)
            {
                destinationSettings.setAdapterSettings(adapterSettings);
            }
            else
            {
                // {ADAPTER_ELEMENT} not found for reference '{ref}' in destination '{destId}'.
                ConfigurationException ex = new ConfigurationException();
                ex.setMessage(REF_NOT_FOUND_IN_DEST, new Object[]{ADAPTER_ELEMENT, ref, destId});
                throw ex;
            }
        }
        else
        {
            //Invalid {ADAPTER_ELEMENT} ref '{ref}' in destination '{destId}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_REF_IN_DEST, new Object[]{ADAPTER_ELEMENT, ref, destId});
            throw ex;
        }
    }

    private void logging(Node root)
    {
        Node logging = selectSingleNode(root, LOGGING_ELEMENT);
        if (logging != null)
        {
            // Validation
            allowedAttributesOrElements(logging, LOGGING_CHILDREN);

            LoggingSettings settings = new LoggingSettings();

            // Log Properties
            NodeList properties = selectNodeList(logging, PROPERTIES_ELEMENT + "/*");
            if (properties.getLength() > 0)
            {
                ConfigMap map = properties(properties, getSourceFileOf(logging));
                settings.addProperties(map);
            }

            NodeList targets = selectNodeList(logging, TARGET_ELEMENT);
            for (int i = 0; i < targets.getLength(); i++)
            {
                Node targetNode = targets.item(i);

                // Target Validation
                requiredAttributesOrElements(targetNode, TARGET_REQ_CHILDREN);
                allowedAttributesOrElements(targetNode, TARGET_CHILDREN);

                String className = getAttributeOrChildElement(targetNode, CLASS_ATTR);

                if (className.length() > 0)
                {
                    TargetSettings targetSettings = new TargetSettings(className);
                    String targetLevel = getAttributeOrChildElement(targetNode, LEVEL_ATTR);

                    if (targetLevel.length() > 0)
                        targetSettings.setLevel(targetLevel);

                    // Filters
                    Node filtersNode = selectSingleNode(targetNode, FILTERS_ELEMENT);
                    if (filtersNode != null)
                    {
                        allowedChildElements(filtersNode, FILTERS_CHILDREN);
                        NodeList filters = selectNodeList(filtersNode, PATTERN_ELEMENT);
                        for (int f = 0; f < filters.getLength(); f++)
                        {
                            Node pattern = filters.item(f);
                            String filter = evaluateExpression(pattern, ".").toString().trim();
                            targetSettings.addFilter(filter);
                        }
                    }

                    // Target Properties
                    properties = selectNodeList(targetNode, PROPERTIES_ELEMENT + "/*");
                    if (properties.getLength() > 0)
                    {
                        ConfigMap map = properties(properties, getSourceFileOf(targetNode));
                        targetSettings.addProperties(map);
                    }

                    settings.addTarget(targetSettings);
                }
            }

            config.setLoggingSettings(settings);
        }
    }

    private void system(Node root)
    {
        Node system = selectSingleNode(root, SYSTEM_ELEMENT);
        if (system != null)
        {
            // Validation
            allowedAttributesOrElements(system, SYSTEM_CHILDREN);

            SystemSettings settings = new SystemSettings();

            // Locale
            Node localeNode = selectSingleNode(system, LOCALE_ELEMENT);

            if (localeNode != null)
            {
                allowedAttributesOrElements(localeNode, LOCALE_CHILDREN);
                String defaultLocaleString = getAttributeOrChildElement(localeNode, DEFAULT_LOCALE_ELEMENT);
                Locale defaultLocale;
                if (defaultLocaleString.length() > 0)
                    defaultLocale = LocaleUtils.buildLocale(defaultLocaleString);
                else
                    defaultLocale = LocaleUtils.buildLocale(null);

                settings.setDefaultLocale(defaultLocale);
            }

            // Manageable
            String manageable = getAttributeOrChildElement(system, MANAGEABLE_ELEMENT);
            settings.setManageable(manageable);

            // Redeploy
            Node redeployNode = selectSingleNode(system, REDEPLOY_ELEMENT);

            if (redeployNode != null)
            {
                allowedAttributesOrElements(redeployNode, REDEPLOY_CHILDREN);

                String enabled = getAttributeOrChildElement(redeployNode, ENABLED_ELEMENT);
                settings.setRedeployEnabled(enabled);

                String interval = getAttributeOrChildElement(redeployNode, WATCH_INTERVAL_ELEMENT);
                if (interval.length() > 0)
                {
                    settings.setWatchInterval(interval);
                }

                NodeList watches = selectNodeList(redeployNode, WATCH_FILE_ELEMENT);
                for (int i = 0; i < watches.getLength(); i++)
                {
                    Node watchNode = watches.item(i);
                    String watch = evaluateExpression(watchNode, ".").toString().trim();
                    if (watch.length() > 0)
                    {
                        settings.addWatchFile(watch);
                    }
                }

                NodeList touches = selectNodeList(redeployNode, TOUCH_FILE_ELEMENT);
                for (int i = 0; i < touches.getLength(); i++)
                {
                    Node touchNode = touches.item(i);
                    String touch = evaluateExpression(touchNode, ".").toString().trim();
                    if (touch.length() > 0)
                    {
                        settings.addTouchFile(touch);
                    }
                }
            }

            ((MessagingConfiguration)config).setSystemSettings(settings);
        }
        else
        {
            // Create a default instance of SystemSettings which by default has setManagable as true
            // and has setRedeployEnabled as false.
            ((MessagingConfiguration)config).setSystemSettings(new SystemSettings());
        }
    }

    private void flexClient(Node root)
    {
        Node flexClient = selectSingleNode(root, FLEX_CLIENT_ELEMENT);
        if (flexClient != null)
        {
            FlexClientSettings flexClientSettings = new FlexClientSettings();
            // Timeout
            String timeout = getAttributeOrChildElement(flexClient, FLEX_CLIENT_TIMEOUT_MINUTES_ELEMENT);
            if (timeout.length() > 0)
            {
                try
                {
                    long timeoutMinutes = Long.parseLong(timeout);
                    if (timeoutMinutes < 0)
                    {
                        // Invalid timeout minutes value ''{0}'' in the <flex-client> configuration section. Please specify a positive value or leave the element undefined in which case flex client instances on the server will be timed out when all associated sessions/connections have shut down.
                        ConfigurationException e = new ConfigurationException();
                        e.setMessage(INVALID_FLEX_CLIENT_TIMEOUT, new Object[]{timeout});
                        throw e;
                    }
                    flexClientSettings.setTimeoutMinutes(timeoutMinutes);
                }
                catch (NumberFormatException nfe)
                {
                    // Invalid timeout minutes value ''{0}'' in the <flex-client> configuration section. Please specify a positive value or leave the element undefined in which case flex client instances on the server will be timed out when all associated sessions/connections have shut down.
                    ConfigurationException e = new ConfigurationException();
                    e.setMessage(INVALID_FLEX_CLIENT_TIMEOUT, new Object[]{timeout});
                    throw e;
                }
            }
            else
            {
                flexClientSettings.setTimeoutMinutes(0); // Default to 0; in this case FlexClients are invalidated when all associated sessions have been invalidated.
            }

            ((MessagingConfiguration)config).setFlexClientSettings(flexClientSettings);
        }
    }

    private void factories(Node root)
    {
        Node factories = selectSingleNode(root, FACTORIES_ELEMENT);
        if (factories != null)
        {
            // Validation
            allowedAttributesOrElements(factories, FACTORIES_CHILDREN);

            NodeList factoryList = selectNodeList(factories, FACTORY_ELEMENT);
            for (int i = 0; i < factoryList.getLength(); i++)
            {
                Node factory = factoryList.item(i);
                factory(factory);
            }
        }
    }

    private void factory(Node factory)
    {
        // Validation
        requiredAttributesOrElements(factory, FACTORY_REQ_CHILDREN);

        String id = getAttributeOrChildElement(factory, ID_ATTR);
        String className = getAttributeOrChildElement(factory, CLASS_ATTR);
        if (isValidID(id))
        {
            FactorySettings factorySettings = new FactorySettings(id, className);

            // Factory Properties
            NodeList properties = selectNodeList(factory, PROPERTIES_ELEMENT + "/*");
            if (properties.getLength() > 0)
            {
                ConfigMap map = properties(properties, getSourceFileOf(factory));
                factorySettings.addProperties(map);
            }
            ((MessagingConfiguration)config).addFactorySettings(id, factorySettings);
        }
        else
        {
            // Invalid {FACTORY_ELEMENT} id '{id}'.
            ConfigurationException ex = new ConfigurationException();
            ex.setMessage(INVALID_ID, new Object[]{FACTORY_ELEMENT, id});
            ex.setDetails(INVALID_ID);
            throw ex;
        }
    }
}
